import os
import logging

from unittest import mock

import yp.client
from infra.deploy_notifications_controller.lib import config, exclusiveservice, http
from infra.deploy_notifications_controller.lib import connection_pool, yp_client, looper
from infra.deploy_notifications_controller.lib import infra_client, qnotifier_client, paste_client, jns_client
from infra.deploy_notifications_controller.lib.models.url_formatter import UrlFormatter
from infra.skylib.debug_server.reverse import ReplServer

log = logging.getLogger('dn')


class Application:
    def __init__(self, cfg, testing_mode=False):
        self.cfg = cfg

        repl_path = config.get_value(cfg, 'debug.sock_path')
        if repl_path:
            self.repl_server = ReplServer(sock_path=repl_path)
            self.repl_server.start()
        else:
            self.repl_server = None

        port = int(config.get_value(cfg, 'http.port', 80))
        self.statistics = http.StatisticsServer(port=port)

        token = config.get_value(cfg, 'yp.token', os.getenv('YP_TOKEN'))

        cluster = config.get_value(cfg, 'yp.cluster')
        address = config.get_value(cfg, 'yp.address').format(cluster=cluster)

        def yp_client_constructor():
            yp_client_base = yp.client.YpClient(
                address=address,
                config={
                    'token': token,
                    'body_log_size_limit': 1,
                },
            )
            return yp_client_base.create_grpc_object_stub()

        yp_stub_pool = connection_pool.ConnectionPool(yp_client_constructor).make_proxy()
        yp_dn_client = yp_client.YpClient(
            yp_stub_pool,
            stage_filter=config.get_value(cfg, 'yp.stage_filter'),
            thread_pool_size=int(config.get_value(cfg, 'yp.thread_pool', 10)),
        )

        if testing_mode:
            mock._CallList = FakeCallList  # otherwise Mock objects save all calls and waste memory
            mock._Call = lambda *args, **kwargs: None
            infra_api_client = mock.AsyncMock(infra_client.InfraClient)
            qnotifier_api_client = mock.AsyncMock(qnotifier_client.QnotifierClient)
            paste_api_client = mock.AsyncMock(paste_client.PasteClient)
            paste_api_client.paste.return_value = 'http://' + ('x' * 80)
            yp_dn_client.save_stage_state = mock.AsyncMock(yp_dn_client.save_stage_state)
            jns_api_client = mock.AsyncMock(jns_client.JNSClient)
        else:
            infra_api_client = infra_client.InfraClient(
                username=config.get_value(cfg, 'infra.username'),
                api_url=config.get_value(cfg, 'infra.api_url'),
                tvm_id=config.get_value(cfg, 'infra.src_app'),
                tvm_secret=config.get_value(cfg, 'infra.secret'),
                dest_tvm_id=config.get_value(cfg, 'infra.dst_app'),
            )

            qnotifier_api_client = qnotifier_client.QnotifierClient(
                api_url=config.get_value(cfg, 'qnotifier.api_url'),
                tvm_id=config.get_value(cfg, 'qnotifier.src_app'),
                tvm_secret=config.get_value(cfg, 'qnotifier.secret'),
                dest_tvm_id=config.get_value(cfg, 'qnotifier.dst_app'),
            )

            paste_api_client = paste_client.PasteClient(
                api_url=config.get_value(cfg, 'paste.api_url'),
                token=config.get_value(cfg, 'paste.token'),
            )

            jns_api_client = jns_client.JNSClient(
                api_url=config.get_value(cfg, 'jns.api_url'),
                token=config.get_value(cfg, 'jns.token'),
                project=config.get_value(cfg, 'jns.project'),
                template=config.get_value(cfg, 'jns.template'),
                retry_timeout=config.get_value(cfg, 'jns.retry_timeout')
            )

        stage_controller = config.get_value(cfg, 'url.stage.controller')
        stage_url_format = config.get_value(cfg, 'url.stage.format')
        user_url_format = config.get_value(cfg, 'url.user.format')

        url_formatter = UrlFormatter(
            stage_format=stage_url_format,
            user_format=user_url_format,
            stage_controller=stage_controller,
        )

        self.looper = looper.Looper(
            yp_client=yp_dn_client,
            infra_client=infra_api_client,
            infra_attempts_delay=float(config.get_value(cfg, 'infra.attempts_delay')),
            qnotifier_client=qnotifier_api_client,
            paste_client=paste_api_client,
            batch_size=int(config.get_value(cfg, 'yp.poll.batch_size')),
            stage_read_period=int(config.get_value(cfg, 'yp.poll.stage_read_period')),
            timestamp_generate_period=int(config.get_value(cfg, 'yp.poll.timestamp_generate_period')),
            qnotifier_send_attempts=int(config.get_value(cfg, 'qnotifier.send_attempts')),
            qnotifier_aggregation_period=int(config.get_value(cfg, 'qnotifier.aggregation_period')),
            history_workers=int(config.get_value(cfg, 'looper.history_workers', 3)),
            history_read_delay=int(config.get_value(cfg, 'looper.history_read_delay', 300)),
            statistics=self.statistics,
            url_formatter=url_formatter,
            jns_client=jns_api_client,
            jns_attempts_delay=float(config.get_value(cfg, 'jns.attempts_delay')),
        )

        coord_cfg = config.get_value(cfg, 'coord')
        acquire_timeout = coord_cfg.get('transaction_timeout')
        if testing_mode:
            self.service = self.looper
        else:
            self.service = exclusiveservice.ExclusiveService(
                coord_cfg,
                f'dn-{coord_cfg["proxy"]}',
                self.looper,
                acquire_timeout_strategy=lambda timeout=acquire_timeout: timeout,
            )

    # Public methods

    async def run(self):
        """
        Start application.
        Blocks until stop was called.
        """
        log.info('starting service...')
        await self.statistics.start()
        await self.service.run()

    async def stop(self):
        """
        Gracefully stop application.
        Can block for a long time or throw exception, be ready.
        """
        log.info('stopping service...')
        await self.service.stop()
        await self.statistics.shutdown()
        log.info('=' * 30)
        if self.repl_server:
            self.repl_server.stop_repl()


class FakeCallList:
    __slots__ = []
    __init__ = lambda self, *args, **kwargs: None
    append = lambda self, *args, **kwargs: None
