import asyncio

from typing import Optional, List, Generic, TypeVar

from infra.deploy_notifications_controller.lib.http import StatisticsServer

T = TypeVar('T')


class AsyncQueueWithStatistic(Generic[T]):
    __slots__ = ['queue', 'statistic_prefix', 'statistics_server']

    queue: asyncio.Queue
    statistic_prefix: str
    statistics_server: StatisticsServer

    def __init__(
        self,
        statistic_prefix: str,
        statistics_server: StatisticsServer
    ):
        self.queue = asyncio.Queue()
        self.statistic_prefix = statistic_prefix
        self.statistics_server = statistics_server

    def update_statistic(self):
        self.statistics_server.set_absolute(f'{self.statistic_prefix}_queue_size', self.queue.qsize())

    def qsize(self):
        return self.queue.qsize()

    def __len__(self):
        return self.qsize()

    def empty(self):
        return self.queue.empty()

    async def put(self, item: T):
        await self.queue.put(item)
        self.update_statistic()

    def put_nowait(self, item: T):
        self.queue.put_nowait(item)
        self.update_statistic()

    def put_all_nowait(self, items: List[T]):
        for item in items:
            self.queue.put_nowait(item)

        self.update_statistic()

    def __contains__(self, item: T):
        # FIXME temporary hack for not duplicating items in queue
        return item in self.queue._queue

    async def get(self) -> T:
        item = await self.queue.get()
        self.update_statistic()
        return item

    def get_nowait(self) -> T:
        # dirty hack to skip waking up putters and thus avoid context switching
        item = self.queue._get()
        self.update_statistic()
        return item

    async def get_with_pending(
        self,
        has_pending_items: bool,
        timeout: Optional[float] = None,
    ) -> Optional[T]:
        if not self.empty():
            return self.get_nowait()

        if has_pending_items:
            assert timeout is not None
            try:
                return await asyncio.wait_for(
                    self.get(),
                    timeout=timeout,
                )
            except asyncio.TimeoutError:
                return None
        else:
            return await self.get()
