import typing
import functools
import threading


Connection = typing.TypeVar('Connection')


class ConnectionPool(typing.Generic[Connection]):
    """
    Connection pool that creates new connections when no free ones exist.
    It is intentionally made as simple as possible and never closes
    connections as its main use are YP connections, and they create
    one class, but use another (stub). Thus we cannot easily close it
    and it's easier to just let 'em lay up here, because it makes no harm.
    """

    _pool: typing.List[Connection]

    def __init__(
        self,
        factory: typing.Callable[[], Connection],
    ):
        self._pool = []
        self._factory = factory
        self._lock = threading.Lock()
        self._connections = 0
        self._thread_conn = threading.local()

    def __enter__(self) -> Connection:
        if 'connection' in self._thread_conn.__dict__:
            self._thread_conn.usages += 1
            return self._thread_conn['connection']

        if self._pool:
            with self._lock:
                if self._pool:
                    self._thread_conn.__dict__.setdefault('usages', 0)
                    self._thread_conn.usages += 1
                    self._thread_conn.connection = conn = self._pool.pop()
                    return conn

        self._thread_conn.__dict__.setdefault('usages', 0)
        self._thread_conn.usages += 1
        self._thread_conn.connection = conn = self._factory()
        with self._lock:
            self._connections += 1

        return conn

    def __exit__(self, exc_type, exc_val, exc_tb):
        conn = self._thread_conn.connection
        self._thread_conn.usages -= 1
        if self._thread_conn.usages:
            return

        del self._thread_conn.connection
        with self._lock:
            self._pool.append(conn)

    def _apply(self, call: str, *args, **kwargs) -> typing.Any:
        with self as conn:
            return getattr(conn, call)(*args, **kwargs)

    def make_proxy(self):
        with self as conn:
            class ConnectionPoolProxy:
                __slots__ = [m for m in dir(conn) if not m.startswith('_') and callable(getattr(conn, m))]

            proxy = ConnectionPoolProxy()

            for item in dir(conn):
                if item in proxy.__slots__:
                    setattr(proxy, item, functools.partial(self._apply, item))

            return proxy
