import copy
from typing import Any, Optional, Set


class DictWrapper:
    __slots__ = ['__values', '__atomic']

    __values: dict
    __atomic: set

    @staticmethod
    def to_values(dict_or_wrapper):
        if isinstance(dict_or_wrapper, DictWrapper):
            return dict_or_wrapper.__values

        if isinstance(dict_or_wrapper, dict):
            return dict_or_wrapper

        return None

    def __init__(
        self,
        values: Optional[Any] = None,
        atomic: Optional[Set] = None,
    ):
        dict_values = DictWrapper.to_values(values)
        self.__values = copy.deepcopy(dict_values) or dict()
        self.__atomic = atomic or set()

    def pop(self, field_name: str):
        self.__values.pop(field_name)

    def update_field(self, field_name: str, value):
        if value is not None:
            self.__values[field_name] = value

    def get_field(self, field_name: str, default_value: Optional[Any] = None):
        value = self.__values.get(field_name, default_value)
        return default_value if value is None else value

    def update_by(self, other):
        other_values = DictWrapper.to_values(other)

        if other_values is None:
            raise TypeError(f'Other can\'t be converted to values: {type(other).__name__}')

        for key, other_value in other_values.items():
            value = self.get_field(key)

            if isinstance(value, DictWrapper) and key not in self.__atomic:
                value.update_by(other_value)
            else:
                self.update_field(key, other_value)

    def empty(self):
        return len(self.__values) == 0

    def to_dict(self):
        values_dict = dict()
        for key, value in self.__values.items():
            value_dict = value.to_dict() if isinstance(value, DictWrapper) else value
            values_dict[key] = value_dict

        return values_dict

    def __repr__(self):
        return f'{self.__values!r}'

    def __str__(self):
        return f'{self.__values}'

    def __eq__(self, other):
        other_values = DictWrapper.to_values(other)

        if other_values is None:
            raise TypeError(f'Other can\'t be converted to values: {type(other).__name__}')

        return self.__values == other_values
