import time
from collections import Counter

from aiohttp import web


class StatisticsServer:
    def __init__(self, port: int):
        self.metrics = Counter()
        self.app = web.Application()
        self.port = port
        self.app.add_routes([
            web.get('/', self.ping),
            web.get('/unistat', self.handle_unistat),
        ])
        self.runner = None
        self.started = time.monotonic()

    def push(self, name: str, value: int) -> None:
        self.metrics[f'{name}_dmmm'] += value

    def set(self, name: str, value: int) -> None:
        self.metrics[f'{name}_ammm'] = value

    def set_absolute(self, name: str, value: int) -> None:
        self.metrics[f'{name}_axxx'] = value

    def set_absolute_summed(self, name: str, value: int) -> None:
        self.metrics[f'{name}_ammx'] = value

    async def start(self) -> None:
        if self.runner is not None:
            return

        self.runner = web.AppRunner(self.app)
        await self.runner.setup()
        server = web.TCPSite(self.runner, '::', self.port)
        await server.start()

    async def shutdown(self) -> None:
        if self.runner is not None:
            await self.runner.cleanup()

    async def ping(self, _: web.Request) -> web.Response:
        return web.Response(status=200, body=b'Ok', content_type='text/plain')

    async def handle_unistat(self, request: web.Request) -> web.Response:
        response = [
            ('alive_time_avvv', time.monotonic() - self.started),
        ]
        response.extend(self.metrics.items())
        return web.json_response(response)
