import asyncio
import logging
import time
import ujson
from functools import partial
from typing import Optional, List, Tuple

import aiohttp
import ticket_parser2.api.v1 as tp2
from yarl import URL


class InfraClient:
    def __init__(self, username: str, api_url: str, tvm_id: int, tvm_secret: str, dest_tvm_id: int):
        self.username = username
        self.api_url = URL(api_url)
        self.connector = aiohttp.TCPConnector(
            keepalive_timeout=60,
            limit=10,
            ttl_dns_cache=600,
        )
        self.tvm = tp2.TvmClient(tp2.TvmApiClientSettings(
            self_client_id=tvm_id,
            self_secret=tvm_secret,
            dsts={'infra': dest_tvm_id},
        ))
        self.tvm_ticket = None
        self.session = None

    async def renew_ticket(self):
        self.tvm_ticket = await asyncio.get_event_loop().run_in_executor(
            None,
            partial(self.tvm.get_service_ticket_for, "infra")
        )

    async def ticket_renewal_loop(self):
        log = logging.getLogger('infra-tvm')
        while True:
            try:
                await self.renew_ticket()
            except Exception as e:
                log.exception("ticket renew failed: %s", e)
                await asyncio.sleep(5)
            else:
                log.info("TVM ticket renewed")
                await asyncio.sleep(600)

    async def __aenter__(self):
        self.session = aiohttp.ClientSession(
            connector=self.connector,
            headers={
                'Accept': 'application/json',
                'User-Agent': '/infra/deploy_notifications_controller 1.0',
            },
            json_serialize=ujson.dumps,
        )
        return await self.session.__aenter__()

    async def __aexit__(self, *args):
        try:
            await self.session.__aexit__(*args)
        finally:
            self.session = None

    def create_event_api_url(self,
                             event_id: Optional[int] = None):
        event_id_suffix_part = '' if event_id is None else f'/{event_id}'
        url_suffix = f'events{event_id_suffix_part}'

        url = self.api_url.join(URL(url_suffix))

        return url

    async def get_actual_ticket(self):
        if self.tvm_ticket is None:
            await self.renew_ticket()

        return self.tvm_ticket

    async def create_headers(self):
        return {
            'X-Ya-Service-Ticket': await self.get_actual_ticket(),
        }

    async def get_current_events(self,
                                 env_id: int,
                                 meta: dict,
                                 latest_revision: int) -> List[Tuple[int, str, dict]]:
        assert self.session is not None

        headers = await self.create_headers()

        async with self.session.get(
            self.create_event_api_url(),
            headers=headers,
            params={
                'from': str(int(time.time())),
                'environmentId': str(env_id),
                'type': 'maintenance',
            },
        ) as response:
            response.raise_for_status()
            data = await response.json(loads=ujson.loads)

            events = []

            for item in data:
                actual_meta = item.get('meta', {})

                was_requested = (item['created_by'] == self.username or actual_meta.get('reported_by') == self.username)
                for key, expected_value in meta.items():
                    if actual_meta.get(key) != expected_value:
                        was_requested = False

                # processing all events <= latest revision
                event_revision = actual_meta.get('revision', 0)
                was_requested &= event_revision <= latest_revision

                if was_requested:
                    event = (item['id'], item['title'], actual_meta)
                    events.append(event)

            return events

    async def create_event(
        self,
        service_id: int,
        env_id: int,
        title: Optional[str],
        description: Optional[str],
        author: Optional[str] = None,
        meta: Optional[dict] = None,
        start_time: Optional[int] = None,
    ) -> int:
        assert self.session is not None

        headers = await self.create_headers()

        start_time = start_time or int(time.time())
        meta = meta or {}
        meta['reported_by'] = self.username

        json = {
            'title': title,
            'description': description,
            'serviceId': service_id,
            'environmentId': env_id,
            'startTime': start_time,
            'type': 'maintenance',
            'severity': 'minor',
            'setAllAvailableDc': True,
            'sendEmailNotifications': True,
            'meta': meta,
            'reporter': author or self.username,
        }

        async with self.session.post(
            self.create_event_api_url(),
            headers=headers,
            json=json,
        ) as response:
            response.raise_for_status()
            data = await response.json()
            return data['id']

    async def update_event(
        self,
        event_id: int,
        event_title: str,
        meta: dict,
        author: Optional[str],
        finish_time: Optional[int] = None,
    ):
        assert self.session is not None

        headers = await self.create_headers()

        json = {
            'reporter': author or self.username,
            'title': event_title,
            'meta': meta,
        }

        if finish_time is not None:
            json['finishTime'] = finish_time

        async with self.session.put(
            self.create_event_api_url(event_id=event_id),
            headers=headers,
            json=json,
        ) as response:
            response.raise_for_status()
