import time
import uuid

import aiohttp
from yarl import URL


class JNSClient:
    def __init__(self, api_url: str, token: str, project: str, template: str, retry_timeout: int):
        self.api_url = URL(api_url)
        self.jns_project = project
        self.jns_template = template
        self.retry_timeout = retry_timeout
        connector = aiohttp.TCPConnector(
            keepalive_timeout=60,
            limit=10,
            ttl_dns_cache=600,
        )
        self.session = aiohttp.ClientSession(
            connector=connector,
            headers={
                'Content-Type': 'application/json',
                'User-Agent': '/infra/deploy_notifications_controller 1.0',
                'Authorization': f'OAuth {token}',
            },
        )

    async def __aenter__(self):
        return await self.session.__aenter__()

    async def __aexit__(self, *args):
        await self.session.__aexit__(*args)

    async def send_to_channel(self, target_project: str, channel: str, body: str):
        req = {
            'project': self.jns_project,
            'template': self.jns_template,
            'request_id': str(uuid.uuid4()),
            'retry_until': int(time.time()) + self.retry_timeout,
            'params': {
                'body': {
                    'string_value': body
                }
            },
            'target_project': target_project,
            'channel': channel,
        }
        async with self.session.post(
            url=self.api_url.join(URL('messages/send_to_channel')),
            json=req,
        ) as json_rsp:
            json_rsp.raise_for_status()
