from enum import Enum
from typing import List, Optional, Set, Any, Dict
import ujson

from infra.deploy_notifications_controller.lib import difflib  # type: ignore
from infra.deploy_notifications_controller.lib.models.notification_policy import NotificationPolicyAction
from infra.deploy_notifications_controller.lib.qnotifier_client import Header


# NOTE these are not dataclasses because slotted ones are currently not supported in python

class Action:
    __slots__ = ['stage_id', 'timestamp']

    stage_id: str
    timestamp: int  # nanoseconds

    def __init__(self, stage_id: str, timestamp: int):
        self.stage_id = stage_id
        self.timestamp = timestamp

    def __eq__(self, other):
        if not isinstance(other, Action):
            raise TypeError(f'Other is not Action: {type(other).__name__}')

        return (self.stage_id == other.stage_id
                and self.timestamp == other.timestamp)


class InfraChange(Action):
    __slots__ = [
        'service_id',
        'environment_id',
        'event_id',
        'name',
        'description',
        'event_kind',
        'author',
        'revision',
    ]

    class EventKind(Enum):
        STARTED = 'created'
        FINISHED = 'finished'
        CANCELLED = 'cancelled'

    service_id: int
    environment_id: int
    event_kind: EventKind
    name: Optional[str]
    description: Optional[str]
    author: str
    revision: int

    def __init__(
        self,
        stage_id: str,
        timestamp: int,
        service_id: int,
        environment_id: int,
        author: str,
        event_kind: EventKind,
        revision: int,
        name: Optional[str] = None,
        description: Optional[str] = None,
    ):
        super().__init__(stage_id=stage_id, timestamp=timestamp)
        self.service_id = service_id
        self.environment_id = environment_id
        self.event_kind = event_kind
        self.name = name
        self.author = author
        self.description = description
        self.revision = revision

    def __eq__(self, other):
        if not isinstance(other, InfraChange):
            raise TypeError(f'Other is not InfraChange: {type(other).__name__}')

        return (super().__eq__(other)
                and self.service_id == other.service_id
                and self.environment_id == other.environment_id
                and self.event_kind == other.event_kind
                and self.author == other.author
                and self.revision == other.revision)

    def __repr__(self) -> str:
        return (
            f'InfraChange(service_id={self.service_id}, '
            f'environment_id={self.environment_id}, '
            f'event_kind={self.event_kind}, '
            f'author={self.author}, '
            f'revision={self.revision}'
        )

    class EventChange(Enum):
        CREATE = 'create'
        CLOSE = 'close'

    event_kind_to_change = {
        EventKind.STARTED: EventChange.CREATE,
        EventKind.FINISHED: EventChange.CLOSE,
        EventKind.CANCELLED: EventChange.CLOSE,
    }

    @property
    def event_change(self) -> EventChange:
        return InfraChange.event_kind_to_change[self.event_kind]

    @property
    def timestamp_seconds(self):
        return int(self.timestamp / 1e9)

    @property
    def start_time(self) -> Optional[int]:
        if self.event_change == InfraChange.EventChange.CREATE:
            return self.timestamp_seconds

        return None

    @property
    def finish_time(self) -> Optional[int]:
        if self.event_change == InfraChange.EventChange.CLOSE:
            return self.timestamp_seconds

        return None

    def update_title(self, event_title: str) -> str:
        if self.event_kind == InfraChange.EventKind.CANCELLED:
            event_title += " (" + self.event_kind.value + ")"

        return event_title

    def create_meta(
        self,
        event_kind: Optional[EventKind] = None,
        with_revision: bool = True,
    ) -> Dict[str, Any]:
        meta = {
            'stage_id': self.stage_id,
            'event_kind': (event_kind or self.event_kind).value,
        }

        if with_revision:
            meta['revision'] = self.revision

        return meta


class QnotifierMessage(Action):
    __slots__ = ['title', 'plain_text', 'html', 'tags', 'attempts', 'state', 'authors', 'project_id', 'change_kinds', 'revisions']

    SEPARATOR = '\n\u0082'  # separator for cutting messages (if needed)

    title: str
    plain_text: str
    html: str
    tags: List[str]
    attempts: int
    state: Optional[dict]
    authors: Set[str]
    project_id: Optional[str]
    change_kinds: Set[str]
    revisions: Set[int]

    def __init__(
        self,
        stage_id: str,
        timestamp: int,
        title: str,
        plain_text: str,
        html: str,
        tags: List[str],
        attempts: int = 0,
        state: Optional[dict] = None,
        authors: Optional[Set[str]] = None,
        project_id: Optional[str] = None,
        change_kinds: Optional[Set[str]] = None,
        revisions: Optional[Set[int]] = None,
    ):
        super().__init__(stage_id=stage_id, timestamp=timestamp)
        self.title = title
        self.plain_text = plain_text
        self.html = html
        self.tags = tags
        self.attempts = attempts
        self.state = state
        self.authors = authors or set()
        self.project_id = project_id
        self.change_kinds = change_kinds or set()
        self.revisions = revisions or set()

    @property
    def headers(self) -> List[Header]:
        headers = [
            {
                'name': 'X-Qnotifier-Tags',
                'value': ', '.join(self.tags),
            },
            {
                'name': 'X-Deploy',
                'value': 'yes'
            },
            {
                'name': 'X-Deploy-Stage-Id',
                'value': self.stage_id,
            },
            {
                'name': 'X-Deploy-Change-Authors',
                'value': ', '.join(self.authors),
            },
        ]
        if self.project_id:
            headers.append({
                'name': 'X-Deploy-Project-Id',
                'value': self.project_id,
            })
        if self.change_kinds:
            headers.append({
                'name': 'X-Deploy-Change-Statuses',
                'value': ', '.join(self.change_kinds),
            })
        if self.revisions:
            headers.append({
                'name': 'X-Deploy-Change-Revisions',
                'value': ', '.join(str(rev) for rev in sorted(self.revisions)),
            })

        return headers

    @property
    def html_style(self) -> str:
        # well, that's kinda dirty to depend on internal HtmlDiff representation,
        # but we don't want to implement our own html formatter, aren't we?
        return f'<style type="text/css">{difflib.HtmlDiff._styles}</style>'

    def __repr__(self) -> str:
        return f'<QnotifierMessage stage_id={self.stage_id!r} ' \
               f'timestamp={self.timestamp!r} ' \
               f'tags={self.tags!r} ' \
               f'attempts={self.attempts!r} ' \
               f'plain_text={self.plain_text[:100]!r}...>'

    def __str__(self) -> str:
        return f'title="{self.title}" ' \
               f'stage_id={self.stage_id} ' \
               f'timestamp={self.timestamp} ' \
               f'project_id={self.project_id} ' \
               f'revisions={self.revisions} ' \
               f'authors={self.authors} ' \
               f'tags={self.tags} ' \
               f'change_kinds={self.change_kinds} ' \
               f'attempts={self.attempts}'

    def __eq__(self, other):
        if not isinstance(other, QnotifierMessage):
            raise TypeError(f'Other is not QNotifierMessage: {type(other).__name__}')

        return (super().__eq__(other)
                and self.tags == other.tags
                and self.attempts == other.attempts
                and self.state == other.state
                and self.authors == other.authors
                and self.project_id == other.project_id
                and self.change_kinds == other.change_kinds
                and self.revisions == other.revisions)


class DummyQnotifierMessage(QnotifierMessage):
    """
    FIXME: this class is kinda hack
    This message is not a real message to send,
    it's just a marker with timestamp, that should be used to
    bump last_timestamp in stage when no real changes occurred but
    there were some records in history that we skipped and thus we
    don't want to select them next time.
    """
    __slots__ = []

    def __init__(self, stage_id: str, timestamp: int, **kwargs):
        super().__init__(
            stage_id=stage_id,
            timestamp=timestamp,
            title='',
            plain_text='',
            html='',
            tags=[],
            **kwargs
        )

    def __eq__(self, other):
        if not isinstance(other, DummyQnotifierMessage):
            raise TypeError(f'Other is not DummyQnotifierMessage: {type(other).__name__}')

        return super().__eq__(other)

    def __str__(self):
        return f'DummyMessage: stage_id={self.stage_id} timestamp={self.timestamp}'


class Notification(Action):
    __slots__ = ['stage_id', 'timestamp', 'author', 'event_kind', 'revision', 'action', 'deploy_unit_id', 'dynamic_resource_id', 'comment']

    class EventKind(Enum):
        STAGE_DEPLOY_STARTED = 'STAGE DEPLOY STARTED'
        STAGE_DEPLOYED = 'STAGE DEPLOYED'
        DEPLOY_UNIT_STARTED = 'DEPLOY UNIT STARTED'
        DEPLOY_UNIT_FINISHED = 'DEPLOY UNIT FINISHED'
        DYNAMIC_RESOURCE_STARTED = 'DYNAMIC RESOURCE STARTED'
        DYNAMIC_RESOURCE_FINISHED = 'DYNAMIC RESOURCE FINISHED'

    def __init__(
        self,
        stage_id: str,
        timestamp: int,
        author: str,
        event_kind: EventKind,
        revision: int,
        action: NotificationPolicyAction,
        deploy_unit_id: Optional[str] = None,
        dynamic_resource_id: Optional[str] = None,
        comment: Optional[str] = None,
    ):
        super().__init__(stage_id=stage_id, timestamp=timestamp)
        self.author = author
        self.event_kind = event_kind
        self.revision = revision
        self.action = action
        self.deploy_unit_id = deploy_unit_id
        self.dynamic_resource_id = dynamic_resource_id
        self.comment = comment

    def to_dict(self) -> dict:
        result = {
            'author': self.author,
            'event': self.event_kind.value,
            'revision': self.revision,
            'comment': self.comment,
        }
        if self.deploy_unit_id:
            result['deploy_unit_id'] = self.deploy_unit_id
        if self.dynamic_resource_id:
            result['dynamic_resource_id'] = self.dynamic_resource_id
        return result

    def to_json(self) -> str:
        return ujson.dumps(self.to_dict())
