from typing import Optional


class NotificationPolicyAction:
    @classmethod
    def create(cls, values: dict):
        if 'http_action' in values:
            params = values['http_action']
            return HttpAction(url=params['url'], attempts=params.get('attempts'))
        if 'jns_message' in values:
            params = values['jns_message']
            return JNSMessageAction(project=params['project'], channel=params['channel'])


class Rule:
    __slots__ = ['start_actions', 'finish_actions']

    start_actions: list[NotificationPolicyAction]
    finish_actions: list[NotificationPolicyAction]

    def __init__(
        self,
        start_actions: Optional[list[dict]] = None,
        finish_actions: Optional[list[dict]] = None,
    ):
        self.start_actions = [
            NotificationPolicyAction.create(acts)
            for acts in start_actions
        ] if start_actions else []
        self.finish_actions = [
            NotificationPolicyAction.create(acts)
            for acts in finish_actions
        ] if finish_actions else []


class NotificationPolicy:
    __slots__ = ['stage_id', 'stage_actions', 'deploy_unit_actions', 'dynamic_resource_actions']

    stage_id: str
    stage_actions: Optional[Rule]
    deploy_unit_actions: Optional[dict[str, Rule]]
    dynamic_resource_actions: Optional[dict[str, Rule]]

    def __init__(
        self, stage_id: str,
        spec: dict[str, dict],
    ):
        self.stage_id = stage_id
        self.stage_actions = Rule(**spec['stage_actions']) if 'stage_actions' in spec else None
        self.deploy_unit_actions = {
            du_id: Rule(**actions)
            for du_id, actions in spec['deploy_unit_actions'].items()
        } if 'deploy_unit_actions' in spec else {}
        self.dynamic_resource_actions = {
            dr_id: Rule(**actions)
            for dr_id, actions in spec['dynamic_resource_actions'].items()
        } if 'dynamic_resource_actions' in spec else {}


class HttpAction(NotificationPolicyAction):
    __slots__ = ['url', 'attempts', 'attempts_current']

    def __init__(
        self,
        url: str,
        attempts: Optional[int] = 3
    ):
        self.url = url
        self.attempts = attempts
        self.attempts_current = 0


class JNSMessageAction(NotificationPolicyAction):
    __slots__ = ['project', 'channel']

    def __init__(self, project: str, channel: str):
        self.project = project
        self.channel = channel
