from typing import Optional, List, Dict, Any
from infra.deploy_notifications_controller.lib.dict_wrapper import DictWrapper


class Meta(DictWrapper):
    __slots__ = []

    class Field:
        PROJECT_ID = 'project_id'
        ACL = 'acl'
        ID = 'id'
        UUID = 'uuid'

    ATOMIC_FIELDS = {
        Field.PROJECT_ID, Field.ACL
    }

    def __init__(
        self,
        project_id: Optional[str] = None,
        acl: Optional[List[Dict[str, Any]]] = None,
        id: Optional[str] = None,
        uuid: Optional[str] = None,
        **kwargs
    ):
        super().__init__(
            **kwargs,
            atomic=Meta.ATOMIC_FIELDS,
        )

        self.update_field(Meta.Field.PROJECT_ID, project_id)
        self.update_field(Meta.Field.ACL, acl)
        self.update_field(Meta.Field.ID, id)
        self.update_field(Meta.Field.UUID, uuid)

    @property
    def id(self) -> str:
        return self.get_field(Meta.Field.ID)

    @property
    def uuid(self) -> str:
        return self.get_field(Meta.Field.UUID)

    @property
    def acl(self) -> dict:
        return self.get_field(Meta.Field.ACL)

    @property
    def project_id(self) -> str:
        return self.get_field(Meta.Field.PROJECT_ID)


class Spec(DictWrapper):
    __slots__ = []

    class Field:
        REVISION = 'revision'
        REVISION_INFO = 'revision_info'

    def __init__(
        self,
        revision: Optional[int] = None,
        **kwargs
    ):
        super().__init__(**kwargs)
        self.update_field(Spec.Field.REVISION, revision)

    @property
    def revision(self) -> int:
        return self.get_field(Spec.Field.REVISION, 0)

    @property
    def description(self) -> str:
        return self.get_field(Spec.Field.REVISION_INFO, {}).get('description', '')


class Status(DictWrapper):
    __slots__ = []

    def __init__(
        self,
        **kwargs
    ):
        super().__init__(**kwargs)

        self.cleanup()

    def cleanup(self):
        for du in self.get_field('deploy_units', {}).values():
            du.pop('target_spec_timestamp', None)
            du.get('failed', {}).pop('last_transition_time', None)
            du.get('ready', {}).pop('last_transition_time', None)
            du.get('in_progress', {}).pop('last_transition_time', None)
            du.pop('progress', None)
            du.pop('current_target', None)

        for dr in self.get_field('dynamic_resources', {}).values():
            dr.get('status', {}).get('ready', {}).get('condition', {}).pop('last_transition_time', None)
            dr.get('status', {}).get('error', {}).get('condition', {}).pop('last_transition_time', None)
            dr.get('status', {}).get('in_progress', {}).get('condition', {}).pop('last_transition_time', None)
            dr.pop('current_target', None)


class StageState(DictWrapper):
    __slots__ = []

    class Field:
        META = 'meta'
        SPEC = 'spec'
        STATUS = 'status'

    ATOMIC_FIELDS = {Field.SPEC, Field.STATUS}

    def __init__(
        self,
        meta: Optional[Meta] = None,
        spec: Optional[Spec] = None,
        status: Optional[Status] = None,
        **kwargs
    ):
        super().__init__(
            **kwargs,
            atomic=StageState.ATOMIC_FIELDS,
        )

        self.update_field(StageState.Field.META, meta)
        self.update_field(StageState.Field.SPEC, spec)
        self.update_field(StageState.Field.STATUS, status)

    @property
    def meta(self) -> Meta:
        return self.get_field(StageState.Field.META, Meta())

    @property
    def spec(self) -> Spec:
        return self.get_field(StageState.Field.SPEC, Spec())

    @property
    def status(self) -> Status:
        return self.get_field(StageState.Field.STATUS, Status())


class StageStateHolder:
    __slots__ = ['state']

    state: StageState

    def __init__(
        self,
        state: Optional[StageState] = None,
        **kwargs
    ):
        self.state = state or StageState(**kwargs)

    @property
    def meta(self) -> Meta:
        return self.state.meta

    @property
    def spec(self) -> Spec:
        return self.state.spec

    @property
    def status(self) -> Status:
        return self.state.status

    @property
    def id(self) -> str:
        return self.meta.id

    @property
    def uuid(self) -> str:
        return self.meta.uuid

    @property
    def project_id(self) -> str:
        return self.meta.project_id

    @property
    def acl(self) -> dict:
        return self.meta.acl

    @property
    def revision(self) -> int:
        return self.spec.revision

    @property
    def description(self) -> str:
        return self.spec.description

    def __eq__(self, other):
        if not isinstance(other, StageStateHolder):
            raise TypeError(f'Other is not StageStateHolder: {type(other).__name__}')

        return self.state == other.state

    def __str__(self):
        return f'State={self.state}'


class Stage(StageStateHolder):
    __slots__ = [
        'max_timestamp',
        'last_timestamp',
        'infra_service',
        'infra_environment',
    ]

    # NOTE we are storing everything as dict, not proto message.
    # This is to simplify serialization to yaml with the following diff calculation
    # as protobuf MessageDifferencer is not available for python
    state: StageState
    max_timestamp: Optional[int]  # actual timestamp in trunk
    last_timestamp: Optional[int]  # last processed change timestamp
    infra_service: Optional[int]
    infra_environment: Optional[int]

    def __init__(
        self,
        max_timestamp: Optional[int] = None,
        last_timestamp: Optional[int] = None,
        infra_service: Optional[int] = None,
        infra_environment: Optional[int] = None,
        **kwargs
    ):
        super().__init__(**kwargs)
        self.max_timestamp = max_timestamp
        self.last_timestamp = last_timestamp
        self.infra_service = infra_service
        self.infra_environment = infra_environment

    @property
    def update_lag(self) -> int:
        if self.last_timestamp is None or self.max_timestamp is None:
            return 0

        return max(0, self.max_timestamp - self.last_timestamp) // int(1e9)

    @staticmethod
    def history_link(link: str, revision: int):
        return f'{link}/history/{revision}'

    def update_by(self, other):
        other_state = None

        if isinstance(other, StageStateHolder):
            other_state = other.state

        if isinstance(other, StageState):
            other_state = other

        if other_state is None:
            raise TypeError(f'Other is not StageStateHolder or StageState: {type(other).__name__}')

        self.state.update_by(other_state)

    def __eq__(self, other):
        if not isinstance(other, Stage):
            raise TypeError(f'Other is not Stage: {type(other).__name__}')

        return (super().__eq__(other)
                and self.last_timestamp == other.last_timestamp)

    def __str__(self):
        return f'{super().__str__()}, last_timestamp={self.last_timestamp}'
