import textwrap

from typing import Optional

from infra.deploy_notifications_controller.lib.models.action import InfraChange, QnotifierMessage, \
    DummyQnotifierMessage, Notification
from infra.deploy_notifications_controller.lib.models.event_meta import EventMeta
from infra.deploy_notifications_controller.lib.models.notification_policy import NotificationPolicy, \
    NotificationPolicyAction
from infra.deploy_notifications_controller.lib.models.stage import StageState, Meta, Status, Spec, \
    Stage
from infra.deploy_notifications_controller.lib.models.stage_history_change_base import StageHistoryChange, ChangeKind
from infra.deploy_notifications_controller.lib.models.stage_progress import StageProgress
from infra.deploy_notifications_controller.lib.models.url_formatter import UrlFormatter


class StageHistoryCreate(StageHistoryChange):
    __slots__ = []

    def __init__(
        self,
        event: EventMeta,
        meta: Meta,
    ):
        super().__init__(
            state=StageState(meta=meta),
            event=event,
            is_heavy_task=False,
            change_kind=ChangeKind.CREATED,
            need_state=False,
        )

    def create_title(self, base: str):
        return f'{base} in project {self.project_id!r}'

    def process_specific_changes(
        self,
        input_data: StageHistoryChange.InputData,
        output_data: StageHistoryChange.OutputData
    ):
        stage_url = input_data.url_formatter.stage_link(input_data.stage)

        plain_text = textwrap.dedent(
            f"""See its spec here: {stage_url}"""
        )

        html = textwrap.dedent(
            f"""You can see its spec {UrlFormatter.a_href('here', stage_url)}.</div>"""
        )

        output_data.append_text(plain_text, html)


class StageHistoryRemove(StageHistoryChange):
    __slots__ = []

    def __init__(
        self,
        event: EventMeta,
    ):
        super().__init__(
            state=StageState(),
            event=event,
            is_heavy_task=False,
            change_kind=ChangeKind.REMOVED,
            need_state=False,
        )

    def create_title(self, base: str):
        return base

    # TODO now project_id is available
    def process_specific_changes(
        self,
        input_data: StageHistoryChange.InputData,
        output_data: StageHistoryChange.OutputData
    ):
        pass


class StageHistoryUpdate(StageHistoryChange):
    __slots__ = []

    meta: Meta
    spec: Spec
    status: Status

    def __init__(
        self,
        event: EventMeta,
        meta: Optional[Meta] = None,
        spec: Optional[Spec] = None,
        status: Optional[Status] = None,
    ):
        super().__init__(
            state=StageState(
                meta=meta,
                spec=spec,
                status=status,
            ),
            event=event,
            is_heavy_task=True,
            change_kind=ChangeKind.MODIFIED,
            need_state=True,
            title_action='changed',
            tag_action='updated',
        )

    def create_title(self, base: str):
        return base

    @staticmethod
    def du_id_to_int_field(field_name, du_source) -> dict[str, Optional[int]]:
        return {
            du_id: du.get(field_name)
            for du_id, du in du_source.get('deploy_units', {}).items()
        }

    @staticmethod
    def dyn_res_to_int_field(field_name, dyn_res_source) -> dict[str, Optional[int]]:
        return {
            dyn_res_id: dyn_res.get('dynamic_resource').get(field_name)
            for dyn_res_id, dyn_res in dyn_res_source.get('dynamic_resources', {}).items()
        }

    @staticmethod
    def get_progress(spec: Spec, status: Status) -> StageProgress:
        return StageProgress(
            spec_revision=spec.revision,
            du_spec_revisions=StageHistoryUpdate.du_id_to_int_field('revision', spec.to_dict()),
            du_latest_deployed_revisions=StageHistoryUpdate.du_id_to_int_field('latest_deployed_revision', status.to_dict()),
            dyn_res_spec_revisions=StageHistoryUpdate.dyn_res_to_int_field('revision', spec.to_dict()),
            dyn_res_status=status.to_dict().get('dynamic_resources', {}),
        )

    @staticmethod
    def create_change_tag(field_name: str) -> str:
        return f'changed:{field_name}'

    def create_infra_change(
        self,
        revision: int,
        stage: Stage,
        event_kind: InfraChange.EventKind,
    ):
        infra_change = InfraChange(
            stage_id=stage.id,
            timestamp=self.timestamp,
            service_id=stage.infra_service,
            environment_id=stage.infra_environment,
            author=self.author.name,
            event_kind=event_kind,
            revision=revision,
        )

        return infra_change

    def process_specific_changes(
        self,
        input_data: StageHistoryChange.InputData,
        output_data: StageHistoryChange.OutputData
    ):
        stage = input_data.stage
        stage_link = input_data.url_formatter.stage_link(stage)

        update_spec_revision = self.revision or stage.revision

        if self.project_id is not None and self.project_id != stage.project_id:
            plain_text_part = f"Project id changed: {stage.project_id!r} -> {self.project_id!r}."
            html_part = f'<div>Project id changed: <b>{stage.project_id!r}</b> &rarr; <b>{self.project_id!r}</b>.</div>'
            output_data.append_text(plain_text_part, html_part)
            output_data.append_tag(self.create_change_tag('project_id'))
            output_data.revisions.add(update_spec_revision)

        if self.acl is not None:
            plain_diff, html_diff = self.make_diffs(
                stage.acl,
                self.acl,
                'Old ACL',
                'New ACL',
                paste_client=input_data.paste_client,
                loop=input_data.loop,
            )

            if plain_diff.strip():
                output_data.append_text('ACL changed:', '<div>ACL changed:</div>')
                output_data.append_text(plain_diff, html_diff)
                output_data.append_tag(self.create_change_tag('acl'))
                output_data.revisions.add(update_spec_revision)

        if not self.spec.empty():
            plain_diff, html_diff = self.make_diffs(
                stage.spec.to_dict(),
                self.spec.to_dict(),
                f'Revision {stage.revision}',
                f'Revision {self.revision}',
                paste_client=input_data.paste_client,
                loop=input_data.loop,
            )

            if plain_diff.strip():
                revision_url = Stage.history_link(stage_link, self.revision)
                html_revision_url = UrlFormatter.a_href('updated', revision_url)

                plain_text_part = f'Spec was updated: {revision_url}'
                html_part = f'<div>Spec was {html_revision_url}:</div>'

                output_data.append_text(plain_text_part, html_part)
                output_data.append_text(plain_diff, html_diff)
                output_data.append_tag(self.create_change_tag('spec'))
                output_data.revisions.add(update_spec_revision)

        def register_change(
            revision: int,
            event_kind: InfraChange.EventKind,
            change_kind: ChangeKind,
        ) -> InfraChange:
            infra_change = self.create_infra_change(revision, stage, event_kind)
            output_data.register_infra(infra_change)
            output_data.register_change(change_kind)
            output_data.revisions.add(revision)
            return infra_change

        def started(revision: int):
            infra_change = register_change(revision, InfraChange.EventKind.STARTED, ChangeKind.DEPLOYING)

            infra_change.name = f'Stage {stage.id!r}: revision {update_spec_revision}'
            infra_change.description = Stage.history_link(stage_link, update_spec_revision)

        def finished(revision: int, event_kind: InfraChange.EventKind):
            register_change(revision, event_kind, ChangeKind.DEPLOYED)

        if not self.status.empty():
            log = input_data.log

            was_deploying = self.get_progress(stage.spec, stage.status)
            update_spec = self.spec if not self.spec.empty() else stage.spec
            is_deploying = self.get_progress(update_spec, self.status)

            if stage.infra_service and stage.infra_environment:
                log.debug('[%s] was_deploying=%r, is_deploying=%r', stage.id, was_deploying, is_deploying)

                if StageProgress.is_started(was_deploying, is_deploying):
                    finished(is_deploying.spec_revision - 1, InfraChange.EventKind.CANCELLED)
                    if is_deploying:
                        started(is_deploying.spec_revision)
                    else:
                        # DEPLOY-4595
                        pass

                if StageProgress.is_finished(was_deploying, is_deploying):
                    finished(is_deploying.spec_revision, InfraChange.EventKind.FINISHED)
            else:
                log.debug("[%s] has no infra service/environment", stage.id)

            if input_data.notification_policy:
                notifications = self.make_notifications(
                    stage=stage,
                    np=input_data.notification_policy,
                    was_deploying=was_deploying,
                    is_deploying=is_deploying,
                )
                for n in notifications:
                    output_data.notifications.append(n)

    def make_notifications(
        self,
        stage: Stage,
        np: NotificationPolicy,
        was_deploying: StageProgress,
        is_deploying: StageProgress,
    ):

        def create_notification(
            event_kind: Notification.EventKind,
            action: NotificationPolicyAction,
            revision: int,
            du_id: Optional[str] = None,
            dr_id: Optional[str] = None,
        ):
            return Notification(
                stage_id=stage.id,
                timestamp=self.timestamp,
                author=self.author.name,
                event_kind=event_kind,
                revision=revision,
                action=action,
                deploy_unit_id=du_id,
                dynamic_resource_id=dr_id,
                comment=self.description or stage.description
            )

        revision = stage.revision
        new_revision = self.revision or stage.revision
        events = Notification.EventKind
        if np.stage_actions:
            stage_deploy_started = StageProgress.is_started(was_deploying, is_deploying)
            stage_deploy_finished = StageProgress.is_finished(was_deploying, is_deploying)
            if stage_deploy_started:
                for action in np.stage_actions.start_actions:
                    yield create_notification(events.STAGE_DEPLOY_STARTED, action, new_revision)
            if stage_deploy_finished:
                for action in np.stage_actions.finish_actions:
                    yield create_notification(events.STAGE_DEPLOYED, action, revision)

        for du_id, du_rule in np.deploy_unit_actions.items():
            du_deploy_started = StageProgress.is_deploy_unit_started(du_id, was_deploying, is_deploying)
            du_deploy_finished = StageProgress.is_deploy_unit_finished(du_id, was_deploying, is_deploying)
            if du_deploy_started:
                for action in du_rule.start_actions:
                    yield create_notification(events.DEPLOY_UNIT_STARTED, action, new_revision, du_id=du_id)
            if du_deploy_finished:
                for action in du_rule.finish_actions:
                    yield create_notification(events.DEPLOY_UNIT_FINISHED, action, revision, du_id=du_id)

        for dr_id, dr_rule in np.dynamic_resource_actions.items():
            dr_deploy_started = StageProgress.is_dynamic_resource_started(dr_id, was_deploying, is_deploying)
            dr_deploy_finished = StageProgress.is_dynamic_resource_finished(dr_id, was_deploying, is_deploying)
            if dr_deploy_started:
                for action in dr_rule.start_actions:
                    yield create_notification(events.DYNAMIC_RESOURCE_STARTED, action, new_revision, dr_id=dr_id)
            if dr_deploy_finished:
                for action in dr_rule.finish_actions:
                    yield create_notification(events.DYNAMIC_RESOURCE_FINISHED, action, revision, dr_id=dr_id)

    def create_message(
        self,
        input_data: StageHistoryChange.InputData,
        output_data: StageHistoryChange.OutputData
    ) -> Optional[QnotifierMessage]:
        if len(output_data.plain_text_parts) > 1:
            return super().create_message(input_data, output_data)

        stage = input_data.stage

        if self.timestamp > stage.last_timestamp:
            return DummyQnotifierMessage(
                stage_id=stage.id,
                timestamp=self.timestamp,
            )
        else:
            return None
