import asyncio
import textwrap

from enum import Enum
from logging import Logger
from typing import List, Optional, Set, Tuple

from infra.deploy_notifications_controller.lib import difflib  # type: ignore
from infra.deploy_notifications_controller.lib.models.action import InfraChange, QnotifierMessage, \
    Notification
from infra.deploy_notifications_controller.lib.models.event_meta import EventMeta
from infra.deploy_notifications_controller.lib.models.notification_policy import NotificationPolicy
from infra.deploy_notifications_controller.lib.models.stage import StageStateHolder, StageState, Stage
from infra.deploy_notifications_controller.lib.paste_client import PasteClient
from infra.deploy_notifications_controller.lib.models.url_formatter import UrlFormatter
from infra.deploy_notifications_controller.lib.yputil import dict_to_yaml


class ChangeKind(Enum):
    CREATED = 'created'
    MODIFIED = 'modified'
    REMOVED = 'removed'
    DEPLOYING = 'deploying'
    DEPLOYED = 'deployed'


class StageHistoryChange(StageStateHolder):
    __slots__ = ['event', 'is_heavy_task', 'change_kind', 'title_action', 'tag_action', 'need_state']

    class InputData:
        __slots__ = ['stage', 'log', 'url_formatter', 'paste_client', 'loop', 'notification_policy']

        stage: Stage
        log: Optional[Logger]
        url_formatter: Optional[UrlFormatter]
        paste_client: Optional[PasteClient]
        loop: Optional[asyncio.AbstractEventLoop]
        notification_policy: Optional[NotificationPolicy]

        def __init__(
            self,
            stage: Stage,
            log: Optional[Logger] = None,
            url_formatter: Optional[UrlFormatter] = None,
            paste_client: Optional[PasteClient] = None,
            loop: Optional[asyncio.AbstractEventLoop] = None,
            notification_policy: Optional[NotificationPolicy] = None,
        ):
            self.stage = stage
            self.log = log
            self.url_formatter = url_formatter
            self.paste_client = paste_client
            self.loop = loop or asyncio.get_event_loop()
            self.notification_policy = notification_policy

    class OutputData:
        __slots__ = ['plain_text_parts', 'html_parts', 'tags', 'change_kinds', 'infra_changes', 'revisions',
                     'notifications']

        plain_text_parts: List[str]
        html_parts: List[str]
        tags: List[str]
        change_kinds: Set[str]
        infra_changes: List[InfraChange]
        revisions: Set[int]
        notifications: list[Notification]

        def __init__(self):
            self.plain_text_parts = []
            self.html_parts = []
            self.tags = []
            self.change_kinds = set()
            self.infra_changes = []
            self.revisions = set()
            self.notifications = []

        def append_text(self, plain_text_part: str, html_part: str):
            self.plain_text_parts.append(plain_text_part)
            self.html_parts.append(html_part)

        def append_tag(self, tag: str):
            self.tags.append(tag)

        def register_change(self, change_kind: ChangeKind):
            self.change_kinds.add(change_kind.value)

        def register_infra(self, infra_change: InfraChange):
            self.infra_changes.append(infra_change)

    event: EventMeta
    is_heavy_task: bool
    change_kind: ChangeKind
    title_action: str
    tag_action: str
    need_state: bool

    def __init__(
        self,
        event: EventMeta,
        is_heavy_task: bool,
        change_kind: ChangeKind,
        need_state: bool,
        title_action: Optional[str] = None,
        tag_action: Optional[str] = None,
        state: Optional[StageState] = None,
    ):
        super().__init__(state=state)

        self.event = event
        self.is_heavy_task = is_heavy_task
        self.change_kind = change_kind
        self.need_state = need_state
        self.title_action = self.action_name if title_action is None else title_action
        self.tag_action = self.action_name if tag_action is None else tag_action

    @property
    def action_name(self) -> str:
        return self.change_kind.value

    def create_title(self, base: str):
        raise NotImplementedError(f'create_title not implemented in {type(self).__name__}')

    @property
    def timestamp(self):
        return self.event.timestamp

    @property
    def str_time(self):
        return self.event.str_time

    @property
    def author(self):
        return self.event.author

    @staticmethod
    def make_diffs(
        old_dict: dict,
        new_dict: dict,
        old_name: str,
        new_name: str,
        paste_client: Optional[PasteClient] = None,
        loop: Optional[asyncio.AbstractEventLoop] = None,
    ) -> Tuple[str, str]:
        yaml_old = dict_to_yaml(old_dict).splitlines(keepends=True)
        yaml_new = dict_to_yaml(new_dict).splitlines(keepends=True)

        plain_text = ''.join(difflib.unified_diff(
            yaml_old,
            yaml_new,
            fromfile=old_name,
            tofile=new_name,
        )).strip()
        if plain_text and paste_client and loop:
            try:
                diff_link = asyncio.run_coroutine_threadsafe(paste_client.paste(plain_text), loop).result()
            except paste_client.PasteException as e:
                if e.http_code != 200:  # paste likes this error-code when in fact forbids message :(
                    raise
            else:
                plain_text = f'Diff: {diff_link}'

        html = difflib.HtmlDiff(tabsize=2).make_table(
            yaml_old,
            yaml_new,
            fromdesc=old_name,
            todesc=new_name,
            context=True,
        ) + difflib.HtmlDiff._legend

        return plain_text, html

    def process_changes(
        self,
        input_data: InputData,
    ) -> Tuple[Optional[QnotifierMessage], OutputData]:
        output_data = self.OutputData()

        # TODO escape stage id
        # TODO escape project ids

        stage = input_data.stage

        event_time = self.str_time
        author_link = input_data.url_formatter.user_link(self.author)

        main_plain_text_part = textwrap.dedent(
            f"""Stage {stage.id!r} was {self.action_name} at {event_time} by {self.author}@."""
        )

        main_html_part = textwrap.dedent(
            f"""<div>Stage <b>{stage.id!r}</b> was {self.action_name} at <time>{event_time}</time> by {author_link}.<br/>"""
        )

        output_data.append_text(main_plain_text_part, main_html_part)

        self.process_specific_changes(input_data, output_data)

        message = self.create_message(input_data, output_data)

        return message, output_data

    def process_specific_changes(
        self,
        input_data: InputData,
        output_data: OutputData
    ):
        pass

    def create_tags(self, input_data: InputData, output_data: OutputData):
        stage = input_data.stage

        tags = [
            'ya.deploy',
            f'stage:{self.tag_action}',
            f'stage:id:{stage.id}',
            f'stage:uuid:{stage.uuid}',
            f'author:{self.author.name}',
        ]

        for tag in output_data.tags:
            tags.append(tag)

        return tags

    def create_message(
        self,
        input_data: InputData,
        output_data: OutputData
    ) -> Optional[QnotifierMessage]:
        stage = input_data.stage

        title = self.create_title(f"Stage {stage.id!r} {self.title_action}")

        plain_text = QnotifierMessage.SEPARATOR.join(output_data.plain_text_parts)
        html = QnotifierMessage.SEPARATOR.join(output_data.html_parts)

        tags = self.create_tags(input_data, output_data)

        output_data.register_change(self.change_kind)

        revisions = output_data.revisions
        if len(revisions) == 0:
            revisions = None

        message = QnotifierMessage(
            stage_id=stage.id,
            timestamp=self.timestamp,
            title=title,
            plain_text=plain_text,
            html=html,
            tags=tags,
            authors={self.author.name},
            project_id=stage.project_id,
            change_kinds=output_data.change_kinds,
            revisions=revisions,
        )

        return message

    def update_stage(self, input_data: InputData) -> Optional[dict]:
        stage = input_data.stage

        stage.update_by(self)

        stage.last_timestamp = self.timestamp

        input_data.log.debug("[%s] update_lag now = %s", stage.id, stage.update_lag)

        return stage.state.to_dict() if self.need_state else None

    def __str__(self):
        return f'{type(self).__name__} by {self.author!s} at {self.str_time} ({self.timestamp})'
