from enum import Enum
from typing import Optional


class StageProgress:
    class State(Enum):
        DEPLOYED = 'deployed'
        DEPLOYING = 'deploying'

    state: State
    spec_revision: int
    du_spec_revisions: dict[str, Optional[int]]
    du_latest_deployed_revisions: dict[str, Optional[int]]
    dyn_res_spec_revisions: dict[str, Optional[int]]
    dyn_res_current_revisions: dict[str, Optional[int]]
    dyn_res_ready: dict[str, bool]

    def __init__(
        self,
        spec_revision: int,
        du_spec_revisions: dict[str, Optional[int]],
        du_latest_deployed_revisions: dict[str, Optional[int]],
        dyn_res_spec_revisions: dict[str, Optional[int]],
        dyn_res_status: dict[str, dict],
    ):
        deployed = (du_spec_revisions == du_latest_deployed_revisions)
        self.state = StageProgress.State.DEPLOYED if deployed else StageProgress.State.DEPLOYING

        self.spec_revision = spec_revision
        self.du_spec_revisions = du_spec_revisions
        self.du_latest_deployed_revisions = du_latest_deployed_revisions
        self.dyn_res_spec_revisions = dyn_res_spec_revisions
        self.dyn_res_current_revisions = {}
        self.dyn_res_ready = {}
        for dyn_res_id, status in dyn_res_status.items():
            self.dyn_res_current_revisions[dyn_res_id] = status.get('revision', None)
            self.dyn_res_ready[dyn_res_id] = status.get('ready', {}).get('condition', {}).get('status', False)

    def revisions_changed(self, other: 'StageProgress'):
        return self.du_spec_revisions != other.du_spec_revisions

    def deployed_revisions_changed(self, other: 'StageProgress'):
        return self.du_latest_deployed_revisions != other.du_latest_deployed_revisions

    def du_revision_changed(self, du_id: str, other: 'StageProgress'):
        return self.du_spec_revisions[du_id] != other.du_spec_revisions[du_id]

    def du_deploying(self, du_id: str):
        return self.du_spec_revisions[du_id] != self.du_latest_deployed_revisions[du_id]

    def dyn_res_revision_changed(self, dyn_res_id: str, other: 'StageProgress'):
        return self.dyn_res_spec_revisions[dyn_res_id] != other.dyn_res_spec_revisions[dyn_res_id]

    def dyn_res_deploying(self, dyn_res_id: str):
        return (self.dyn_res_current_revisions[dyn_res_id] != self.dyn_res_spec_revisions[dyn_res_id]
                or not self.dyn_res_ready[dyn_res_id])

    def __bool__(self):
        return self.state == self.State.DEPLOYING

    def __repr__(self):
        return (
            f'StageProgress(state={self.State.__name__}.{self.state.name}, '
            f'spec_revision={self.spec_revision!r}, '
            f'du_spec_revisions={self.du_spec_revisions!r}, '
            f'du_latest_deployed_revisions={self.du_latest_deployed_revisions!r})'
        )

    @staticmethod
    def is_started(was_deploying: 'StageProgress', is_deploying: 'StageProgress'):
        return is_deploying.revisions_changed(was_deploying)

    @staticmethod
    def is_finished(was_deploying: 'StageProgress', is_deploying: 'StageProgress'):
        return is_deploying.deployed_revisions_changed(was_deploying) and not is_deploying

    @staticmethod
    def is_deploy_unit_started(
        deploy_unit_id: str,
        was_deploying: 'StageProgress',
        is_deploying: 'StageProgress',
    ) -> bool:
        return (is_deploying.du_deploying(deploy_unit_id) and
                is_deploying.du_revision_changed(deploy_unit_id, was_deploying))

    @staticmethod
    def is_deploy_unit_finished(
        deploy_unit_id: str,
        was_deploying: 'StageProgress',
        is_deploying: 'StageProgress',
    ) -> bool:
        return was_deploying.du_deploying(deploy_unit_id) and not is_deploying.du_deploying(deploy_unit_id)

    @staticmethod
    def is_dynamic_resource_started(
        dynamic_resource_id: str,
        was_deploying: 'StageProgress',
        is_deploying: 'StageProgress',
    ) -> bool:
        return (is_deploying.dyn_res_deploying(dynamic_resource_id) and
                is_deploying.dyn_res_revision_changed(dynamic_resource_id, was_deploying))

    @staticmethod
    def is_dynamic_resource_finished(
        dynamic_resource_id: str,
        was_deploying: 'StageProgress',
        is_deploying: 'StageProgress',
    ) -> bool:
        return (was_deploying.dyn_res_deploying(dynamic_resource_id)
                and not is_deploying.dyn_res_deploying(dynamic_resource_id))
