import aiohttp

import ujson
from yarl import URL


class PasteClient:
    class PasteException(Exception):
        def __init__(self, *args, http_code=None, **kwargs):
            super().__init__(*args, **kwargs)
            self.http_code = http_code

    def __init__(self, api_url: str, token: str):
        self.api_url = URL(api_url)
        self.session = aiohttp.ClientSession(
            headers={
                'User-Agent': '/infra/deploy_notifications_controller 1.0',
                'Authorization': f'OAuth {token}',
            },
            json_serialize=ujson.dumps,
        )

    async def __aenter__(self):
        return await self.session.__aenter__()

    async def __aexit__(self, *args):
        await self.session.__aexit__(*args)

    async def paste(self, text: str, file_type: str = 'diff') -> str:
        payload = {
            'syntax': file_type,
            'text': text,
        }
        async with self.session.post(
            self.api_url,
            data=payload,
            allow_redirects=False,
        ) as r:
            if r.status != 302:
                msg = await r.read()
                raise self.PasteException(f"Paste failed with {r.status} {r.reason}: {msg}", http_code=r.status)

            url = URL(r.headers['Location'])
            if url.is_absolute():
                return str(url)
            else:
                return str(self.api_url.join(url))
