import logging
import typing
import asyncio
from functools import partial

import ujson
import aiohttp
from yarl import URL
import ticket_parser2.api.v1 as tp2


class Header(typing.TypedDict):
    name: str
    value: str


class QnotifierClient:
    def __init__(self, api_url: str, tvm_id: int, tvm_secret: str, dest_tvm_id: int):
        self.api_url = URL(api_url)
        self.tvm = tp2.TvmClient(tp2.TvmApiClientSettings(
            self_client_id=tvm_id,
            self_secret=tvm_secret,
            dsts={'qnotifier': dest_tvm_id},
        ))
        self.tvm_ticket = None
        connector = aiohttp.TCPConnector(
            keepalive_timeout=60,
            limit=10,
            ttl_dns_cache=600,
        )
        self.session = aiohttp.ClientSession(
            connector=connector,
            headers={
                'Accept': 'application/json',
                'User-Agent': '/infra/deploy_notifications_controller 1.0',
            },
            json_serialize=ujson.dumps
        )

    async def renew_ticket(self):
        self.tvm_ticket = await asyncio.get_event_loop().run_in_executor(
            None,
            partial(self.tvm.get_service_ticket_for, "qnotifier")
        )

    async def ticket_renewal_loop(self):
        log = logging.getLogger('qnotifier-tvm')
        while True:
            try:
                await self.renew_ticket()
            except Exception as e:
                log.exception("ticket renew failed: %s", e)
                await asyncio.sleep(5)
            else:
                log.info("TVM ticket renewed")
                await asyncio.sleep(600)

    async def __aenter__(self):
        return await self.session.__aenter__()

    async def __aexit__(self, *args):
        await self.session.__aexit__(*args)

    async def post_event(
        self,
        title: str,
        plain_text: str,
        html: str,
        tags: typing.List[str],
        headers: typing.Optional[typing.List[Header]] = None,
    ):
        if self.tvm_ticket is None:
            await self.renew_ticket()

        headers = headers or []

        async with self.session.post(
            self.api_url,
            headers={
                'X-Ya-Service-Ticket': self.tvm_ticket,
            },
            json={
                'message': plain_text,
                'tags': tags,
                'extra': {
                    'from_email': 'Ya.Deploy <devnull-noreply@yandex-team.ru>',
                    'subject': title,
                    'message-html': html,
                    'headers': headers,
                },
            },
        ) as response:
            response.raise_for_status()
