import asyncio
import logging
from functools import partial
from typing import Optional

import ticket_parser2.api.v1 as tp2


class TvmTicketRenewer:
    __slots__ = [
        '_dest_name',
        '_tvm',
    ]

    _dest_name: str
    _tvm: tp2.TvmClient

    def __init__(
        self,
        dest_name: str,
        tvm: tp2.TvmClient,
    ):
        self._dest_name = dest_name
        self._tvm = tvm

    @staticmethod
    def create(
        dest_name: str,
        tvm_id: int,
        tvm_secret: str,
        dest_tvm_id: int,
    ):
        tvm = tp2.TvmClient(tp2.TvmApiClientSettings(
            self_client_id=tvm_id,
            self_secret=tvm_secret,
            dsts={dest_name: dest_tvm_id},
        ))

        return TvmTicketRenewer(
            dest_name=dest_name,
            tvm=tvm,
        )

    async def renew_ticket(self) -> str:
        return await asyncio.get_event_loop().run_in_executor(
            None,
            partial(self._tvm.get_service_ticket_for, self._dest_name)
        )

    @property
    def dest_name(self):
        return self._dest_name


class CachedTvmTicketRenewer:
    __slots__ = [
        '_renewer',
        '_ticket',
    ]

    _renewer: TvmTicketRenewer
    _ticket: Optional[str]

    def __init__(
        self,
        renewer: TvmTicketRenewer,
    ):
        self._renewer = renewer
        self._ticket = None

    async def _update_cache(self):
        self._ticket = await self._renewer.renew_ticket()

    async def ticket_renewal_loop(self):
        log = logging.getLogger(f'{self._renewer.dest_name}-tvm')
        while True:
            try:
                await self._update_cache()
            except Exception as e:
                log.exception("ticket renew failed: %s", e)
                await asyncio.sleep(5)
            else:
                log.info("TVM ticket renewed")
                await asyncio.sleep(600)

    async def get_ticket(self):
        if self._ticket is None:
            await self._update_cache()

        return self._ticket
