import functools
import itertools
from typing import Iterable, List, TypeVar, Type

import yaml

from yt.yson import yson_types
import yt_yson_bindings


SafeDumper = getattr(yaml, 'CSafeDumper', yaml.SafeDumper)
SafeDumper.add_representer(yson_types.YsonEntity, SafeDumper.represent_none)
SafeDumper.add_representer(yson_types.YsonBoolean, SafeDumper.represent_bool)
SafeDumper.add_representer(yson_types.YsonInt64, SafeDumper.represent_int)
SafeDumper.add_representer(yson_types.YsonUint64, SafeDumper.represent_int)
SafeDumper.add_representer(yson_types.YsonDouble, SafeDumper.represent_float)
SafeDumper.add_representer(yson_types.YsonString, SafeDumper.represent_binary)
SafeDumper.add_representer(yson_types.YsonUnicode, SafeDumper.represent_str)
SafeDumper.add_representer(yson_types.YsonMap, SafeDumper.represent_dict)
SafeDumper.add_representer(yson_types.YsonList, SafeDumper.represent_list)

T = TypeVar('T')


def yson_to_proto(yson_response, proto_type: Type[T]) -> T:
    return yt_yson_bindings.loads_proto(yson_response.yson, proto_type, skip_unknown_fields=True)


def yson_to_proto_list(yson_response, proto_type: Type[T]) -> List[T]:
    raw = yt_yson_bindings.loads(yson_response.yson)
    if not raw or not isinstance(raw, list):
        return []

    return [
        yt_yson_bindings.loads_proto(yt_yson_bindings.dumps(item), proto_type, skip_unknown_fields=True)
        for item in raw
    ]


def dict_to_yaml(message: dict) -> str:
    return yaml.dump(message, Dumper=SafeDumper, default_flow_style=False)


def chunk(chunk_size: int, iterable: Iterable[T]) -> List[T]:
    return list(itertools.islice(iterable, chunk_size))


def window(iterable: Iterable[T], chunk_size: int) -> Iterable[List[T]]:
    return iter(functools.partial(chunk, chunk_size, iter(iterable)), [])
