from typing import List, Iterator, Tuple, Optional

from infra.deploy_queue_controller.lib.models import Ticket
from infra.deploy_queue_controller.lib.yp_client import YpClient


class Actions:
    cancelled_tickets: List[Ticket]
    committed_tickets: List[Tuple[Ticket, str]]
    waiting_tickets: List[Tuple[Ticket, str]]
    yp_client: YpClient

    def __init__(self, yp_client: YpClient):
        self.yp_client = yp_client
        self.cancelled_tickets = []
        self.committed_tickets = []
        self.waiting_tickets = []

    def cancel_tickets(self, tickets: Iterator[Ticket]) -> None:
        self.cancelled_tickets.extend(ticket for ticket in tickets if ticket.is_pending())

    def commit_ticket(self, ticket: Ticket) -> None:
        self.committed_tickets.append((ticket, ticket.release.spec.title))

    def wait_for_tickets(self, tickets: Iterator[Ticket], target_ticket: Optional[Ticket]) -> None:
        if target_ticket is not None:
            wait_message = f'Ticket is waiting in queue, while ticket={target_ticket.meta.id!r} is being activated'
        else:
            wait_message = 'Ticket is waiting in queue'

        self.waiting_tickets.extend(
            (ticket, wait_message)
            for ticket in tickets
            # FIXME DEPLOY-1944 since we are writing "waits for TTT" in wait action message now,
            # we probably should update it, but only when message changes.
            if not ticket.is_waiting()
        )

    async def perform(self, timestamp: int, batch_size: int) -> None:
        key = lambda ticket: ticket.meta.creation_time
        key2 = lambda tup: tup[0].meta.creation_time
        get_id = lambda ticket: ticket.meta.id
        get_id2 = lambda tup: (tup[0].meta.id, tup[1])

        await self.yp_client.submit_ticket_actions(
            timestamp=timestamp,
            cancels=map(get_id, sorted(self.cancelled_tickets, key=key)),
            commits=map(get_id2, sorted(self.committed_tickets, key=key2)),
            waits=map(get_id2, sorted(self.waiting_tickets, key=key2)),
            batch_size=batch_size,
        )
