import os
import logging

import yp.client
from infra.deploy_queue_controller.lib import config, exclusiveservice, http
from infra.deploy_queue_controller.lib import yp_client, queue_controller
from infra.skylib.debug_server.reverse import ReplServer

log = logging.getLogger('dq')


class Application:
    def __init__(self, cfg):
        self.cfg = cfg

        repl_path = config.get_value(cfg, 'debug.sock_path')
        if repl_path:
            self.repl_server = ReplServer(sock_path=repl_path)
            self.repl_server.start()
        else:
            self.repl_server = None

        port = int(config.get_value(cfg, 'http.port', 80))
        self.statistics = http.StatisticsServer(port=port)

        token = config.get_value(cfg, 'yp.token', os.getenv('YP_TOKEN'))
        yp_client_base = yp.client.YpClient(
            address=config.get_value(cfg, 'yp.address'),
            config={
                'token': token,
                'body_log_size_limit': 1,
            },
        )
        yp_client_stub = yp_client_base.create_grpc_object_stub()
        yp_dq_client = yp_client.YpClient(yp_client_stub)
        self.queue_controller = queue_controller.QueueController(
            yp_client=yp_dq_client,
            batch_size=config.get_value(cfg, 'yp.poll.batch_size'),
            username=config.get_value(cfg, 'yp.username'),
            statistics=self.statistics,
        )

        coord_cfg = config.get_value(cfg, 'coord')
        acquire_timeout = coord_cfg.get('transaction_timeout')
        self.service = exclusiveservice.ExclusiveService(
            coord_cfg,
            f'dq-{coord_cfg["proxy"]}',
            self.queue_controller,
            acquire_timeout_strategy=lambda timeout=acquire_timeout: timeout,
        )

    # Public methods

    async def run(self):
        """
        Start application.
        Blocks until stop was called.
        """
        log.info('starting service...')
        await self.statistics.start()
        await self.service.run()

    async def stop(self):
        """
        Gracefully stop application.
        Can block for a long time or throw exception, be ready.
        """
        log.info('stopping service...')
        await self.service.stop()
        await self.statistics.shutdown()
        log.info('=' * 30)
