try:
    import errno
    import socket
    import logging
    import asyncio
    import argparse

    from infra.deploy_queue_controller.lib import app, config
    from infra.skylib import logger
    from infra.skylib.debug_server.reverse import repl_client
except KeyboardInterrupt:
    # don't bother user with python stack trace
    # if interrupted (by Ctrl+C) during imports
    raise SystemExit(1)


log = logging.getLogger('main')


def parse_args():
    p = argparse.ArgumentParser(description='dynamic resource processor')
    p.add_argument('-c', '--cfg',
                   default='./cfg_default.yml',
                   help='path to service cfg file')
    p.add_argument('-a', '--attach',
                   action='store_true',
                   default=False,
                   help='attach to debug socket',
                   )
    # --console is used to dump logs during debugging with IDE
    # when stdout isn't terminal but we don't want to log to file
    p.add_argument('--console',
                   default=False,
                   action='store_true',
                   help='redirect log to stdout (overrides config)')
    return p.parse_args()


async def async_main():
    application = None
    try:
        with logger.temporary_stdout_logger():
            args = parse_args()
            cfg = config.load(path=args.cfg, env_prefix='DQ')

        if args.attach:
            raise SystemExit(repl_client(config.get_value(cfg, 'debug.sock_path')))

        logger.setup_logging(config.get_value(cfg, 'logging'), args.console)

        application = app.Application(cfg)
        await application.run()
    except SystemExit:
        raise
    except KeyboardInterrupt:
        log.info("got Ctrl-C, exiting")
        exit_code = 0
    except socket.error as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on socket error: code=%r msg=%r", errorcode, e.strerror)
        exit_code = 1
    except IOError as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception(
            "exiting on io error: file=%r code=%r msg=%r",
            e.filename, errorcode, e.strerror
        )
        exit_code = 1
    except Exception:
        log.exception("exiting on unhandled error:")
        exit_code = 1
    else:
        exit_code = 0
    finally:
        if application is not None:
            await application.stop()
    raise SystemExit(exit_code)


def main():
    loop = asyncio.get_event_loop()
    loop.run_until_complete(async_main())
