import logging
from typing import List, Optional
from enum import Enum

from yp import data_model


log = logging.getLogger('models')


# NOTE these are not dataclasses because slotted ones are currently not supported in python

class Release:
    __slots__ = ['meta', 'spec']

    meta: data_model.TReleaseMeta
    spec: data_model.TReleaseSpec

    def __init__(
        self,
        meta: data_model.TReleaseMeta,
        spec: data_model.TReleaseSpec,
    ):
        self.meta = meta
        self.spec = spec


class Stage:
    __slots__ = ['meta', 'spec', 'status']

    meta: data_model.TStageMeta
    spec: data_model.TStageSpec
    status: data_model.TStageStatus

    def __init__(
        self,
        meta: data_model.TStageMeta,
        spec: data_model.TStageSpec,
        status: data_model.TStageStatus,
    ):
        self.meta = meta
        self.spec = spec
        self.status = status

    def is_in_progress(self) -> bool:
        revision = self.spec.revision
        # if revision != self.status.revision:
        #    # log.info(f"stage revision {revision} != status {self.status.revision}")
        #    return True

        if len(self.spec.deploy_units) != len(self.status.deploy_units):
            # log.info("not all deploy units present in status")
            return True

        specs = self.spec.deploy_units
        for du_id, du in self.status.deploy_units.items():
            if (
                du_id not in specs
                or du.target_revision != specs[du_id].revision
                or du.in_progress.status == data_model.CS_TRUE
            ):
                # log.info(f"DU status {du_id!r} revision {du.target_revision=} != spec {specs[du_id].revision=} or {du.in_progress.status=} == {data_model.CS_TRUE}")
                return True

        return False


class Ticket:
    __slots__ = ['meta', 'spec', 'status', 'release', 'stage']

    meta: data_model.TDeployTicketMeta
    spec: data_model.TDeployTicketSpec
    status: data_model.TDeployTicketStatus
    release: Optional[Release]
    stage: Optional[Stage]

    def __init__(
        self,
        meta: data_model.TDeployTicketMeta,
        spec: data_model.TDeployTicketSpec,
        status: data_model.TDeployTicketStatus,
        release: Optional[Release] = None,
        stage: Optional[Stage] = None,
    ):
        self.meta = meta
        self.spec = spec
        self.status = status
        self.release = release
        self.stage = stage

    def __repr__(self) -> str:
        return f'Ticket[id={self.meta.id}]'

    def is_committed(self) -> bool:
        return self.status.action.type == data_model.DPAT_COMMIT

    def is_waiting(self) -> bool:
        return self.status.action.type == data_model.DPAT_WAIT

    def is_on_hold(self) -> bool:
        return self.status.action.type == data_model.DPAT_ON_HOLD

    def is_pending(self) -> bool:
        return self.status.action.type in (data_model.DPAT_NONE, data_model.DPAT_WAIT)

    def is_in_progress(self) -> bool:
        dynamic_resources = []
        has_static_resources = False

        for patch in self.spec.patches.values():
            if patch.WhichOneof('payload') != 'sandbox':
                has_static_resources = True
                continue

            if patch.sandbox.WhichOneof('resource_ref') != 'dynamic':
                has_static_resources = True
                continue

            dynamic_resources.append(patch.sandbox.dynamic.dynamic_resource_id)

        if has_static_resources and self.stage.is_in_progress():
            log.debug("ticket %r has static resources and stage is in progress", self.meta.id)
            return True

        for resource_id in dynamic_resources:
            if resource_id not in self.stage.spec.dynamic_resources:
                log.debug("ticket %r has dr %r and stage has not it", self.meta.id, resource_id)
                # well, it means that ticket cannot be applied,
                # so we could skip this patch and report it as non-active
                # but better to prevent than fail
                return True

            spec_revision = self.stage.spec.dynamic_resources[resource_id].dynamic_resource.revision

            if resource_id not in self.stage.status.dynamic_resources:
                log.debug("ticket %r has dr %r and it is not in stage status", self.meta.id, resource_id)
                return True

            status = self.stage.status.dynamic_resources[resource_id]
            if status.current_target.dynamic_resource.revision != spec_revision:
                log.debug("ticket %r has dr %r and its revision %r != status %r", self.meta.id, resource_id, spec_revision, status.current_target.dynamic_resource.revision)
                return True

            if status.status.ready.condition.status != data_model.CS_TRUE:
                log.debug("ticket %r has dr %r and its condition %r != %r", self.meta.id, resource_id, status.status.ready.condition.status, data_model.CS_TRUE)
                return True

        return False

    def older_than(self, ticket: "Ticket") -> bool:
        return self.meta.creation_time < ticket.meta.creation_time


class ReleaseRule:
    __slots__ = [
        'in_progress',
        'target_ticket',
        'mode',
        'outdated_tickets',
        'waiting_tickets',
        'trunk_deployment_termination_policy',
    ]

    class CommitMode(Enum):
        FIRST = "sequential_commit"
        LAST = "maintain_active_trunk"

    class DeploymentTerminationPolicy(Enum):
        WAIT_FOR_COMPLETION = "wait_for_completion"
        TERMINATE = "terminate"

    in_progress: bool
    mode: CommitMode
    target_ticket: Optional[Ticket]
    outdated_tickets: List[Ticket]
    waiting_tickets: List[Ticket]
    trunk_deployment_termination_policy: DeploymentTerminationPolicy

    def __init__(
        self,
        mode: CommitMode,
        trunk_deployment_termination_policy: DeploymentTerminationPolicy,
        in_progress: bool = False,
        ticket: Optional[Ticket] = None,
    ):
        self.mode = mode
        self.in_progress = in_progress
        self.target_ticket = ticket
        self.outdated_tickets = []
        self.waiting_tickets = []
        self.trunk_deployment_termination_policy = trunk_deployment_termination_policy

    def termination_allowed(self) -> bool:
        return (
            self.mode == self.CommitMode.LAST
            and self.trunk_deployment_termination_policy == self.DeploymentTerminationPolicy.TERMINATE
        )

    def updates_blocked(self) -> bool:
        return self.in_progress and not self.termination_allowed()

    def add_ticket(self, ticket: Ticket) -> None:
        # we have active ticket so we will not activate anything
        if ticket.is_in_progress() and not self.termination_allowed():
            log.debug("%r: release rule is in progress for ticket and its termination is not allowed", ticket.meta.id)
            self.in_progress = True
            self.waiting_tickets.extend(self.outdated_tickets)
            if ticket.is_committed():
                log.debug("%r: ticket is committed already", ticket.meta.id)
                self.target_ticket = ticket
            else:
                log.debug("%r: ticket is waiting", ticket.meta.id)
                self.waiting_tickets.append(ticket)
            self.outdated_tickets.clear()

        # depending on autocommit mode we can mark ticket as a target, skip it or put it in queue
        elif not self.updates_blocked():
            if self.target_ticket is None:
                log.debug("%r: is new (and first so far) candidate for commit", ticket.meta.id)
                self.target_ticket = ticket
                return

            if self.mode == ReleaseRule.CommitMode.FIRST:
                if ticket.older_than(self.target_ticket):
                    log.debug("%r: is new candidate for commit", ticket.meta.id)
                    self.waiting_tickets.append(self.target_ticket)
                    self.target_ticket = ticket
                else:
                    log.debug("%r: waiting, we already have older candidate for commit %r", ticket.meta.id,
                              self.target_ticket.meta.id)
                    self.waiting_tickets.append(ticket)

            elif self.mode == ReleaseRule.CommitMode.LAST:
                if self.target_ticket.older_than(ticket):
                    log.debug("%r: is new candidate for commit", ticket.meta.id)
                    self.outdated_tickets.append(self.target_ticket)
                    self.target_ticket = ticket
                else:
                    log.debug("%r: outdated, we already have newer candidate for commit %r", ticket.meta.id,
                              self.target_ticket.meta.id)
                    self.outdated_tickets.append(ticket)

        # if rule is active, we can just mark all tickets as waiting
        elif ticket.is_pending():
            log.debug("%r: waiting, rule is in progress", ticket.meta.id)
            self.waiting_tickets.append(ticket)
