package ru.yandex.infra.sbr_updater;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;

import javax.servlet.http.HttpServlet;

import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.ImmutableMap;
import com.typesafe.config.Config;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.handler.DefaultHandler;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.thread.QueuedThreadPool;

import ru.yandex.infra.controller.concurrent.CypressLockingService;
import ru.yandex.infra.controller.concurrent.DummyLeaderService;
import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.concurrent.LeaderServiceImpl;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.servlets.UnistatServlet;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.ObjectBuilderDescriptor;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yp.YpInstance;
import ru.yandex.yp.YpRawClient;
import ru.yandex.yp.YpRawClientBuilder;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.Autogen;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;

public class ConfigUtils {
    private static final String SERVICE_NAME = "sbr_updater";
    private static final ObjectBuilderDescriptor<Autogen.TStageMeta, StageMeta> STAGE_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(
                    TStageSpec::newBuilder, TStageStatus::newBuilder, StageMeta::fromProto,
                    Autogen.TStageMeta.getDefaultInstance());


    private static <T> Map<String, T> extractMap(Config config, BiFunction<Config, String, T> extractor) {
        return StreamEx.of(config.root().keySet())
                .toMap(key -> extractor.apply(config, key));
    }

    public static LeaderService leaderService(Config config, MetricRegistry metricRegistry) {
        if (!config.getBoolean("use_lock")) {
            return new DummyLeaderService(metricRegistry);
        }

        return new LeaderServiceImpl(
                SERVICE_NAME,
                new CypressLockingService(
                        config.getString("yt_proxy"),
                        config.getString("yt_token"),
                        YPath.simple(config.getString("lock_path")),
                        YPath.simple(config.getString("epoch_path"))),
                metricRegistry
        );
    }

    public static Map<String, String> labels(Config config) {
        return extractMap(config, Config::getString);
    }

    public static YpRawClient ypRawClient(Config ypConfig) {
        Config instanceConfig = ypConfig.getConfig("master");
        Duration mastersTimeout = ypConfig.getDuration("masters_request_timeout");
        Duration requestTimeout = ypConfig.getDuration("request_timeout");

        YpInstance instance = new YpInstance(instanceConfig.getString("host"), instanceConfig.getInt("port"));
        return new YpRawClientBuilder(instance, () -> ypConfig.getString("token"))
                .setTimeout(requestTimeout.toNanos(), TimeUnit.NANOSECONDS)
                .setMastersRequestTimeout(mastersTimeout.toNanos(), TimeUnit.NANOSECONDS)
                .setUseMasterDiscovery(ypConfig.getBoolean("use_master_discovery"))
                .setUsePlaintext(!ypConfig.getBoolean("secure"))
                .setMaxInboundMessageSize(ypConfig.getBytes("inbound_message_size_limit").intValue())
                .setMaxInboundMetadataSize(ypConfig.getBytes("inbound_metadata_size_limit").intValue())
                .build();
    }

    public static LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository(
            YpRawObjectService objectService, int selectPageSize, int watchPageSize,
            Map<String, String> labels,
            GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.STAGE, labels,
                Optional.empty(),
                objectService, STAGE_DESCRIPTOR, selectPageSize, watchPageSize, gaugeRegistry);
    }

    public static SandboxApi getSandboxApi(Config config) {
        AsyncHttpClient sandboxHttpClient = httpClient(config.getConfig("http_client"));

        return new SandboxApi(sandboxHttpClient,
                config.getString("base_url"),
                config.getString("token")
        );
    }

    private static AsyncHttpClient httpClient(Config config) {
        return new DefaultAsyncHttpClient(new DefaultAsyncHttpClientConfig.Builder()
                .setRequestTimeout((int) config.getDuration("request_timeout").toMillis())
                .build());
    }

    private static Server buildServer(Config config, int port, String address, Map<String, HttpServlet> servlets) {
        int workerThreads = config.getInt("worker_threads");
        Server server = new Server(new QueuedThreadPool(workerThreads, workerThreads));

        int selectorThreads = config.getInt("selector_threads");
        ServerConnector connector = new ServerConnector(server, 0, selectorThreads);
        connector.setReuseAddress(true);
        connector.setHost(address);
        connector.setPort(port);
        connector.setIdleTimeout(config.getDuration("connection_timeout", TimeUnit.MILLISECONDS));
        connector.setAcceptQueueSize(config.getInt("accept_backlog_size"));
        server.addConnector(connector);

        ServletContextHandler context = new ServletContextHandler();
        context.setContextPath("/");
        servlets.forEach((name, servlet) -> context.addServlet(new ServletHolder(servlet), name));
        HandlerCollection handlers = new HandlerCollection();
        handlers.setHandlers(new Handler[]{context, new DefaultHandler()});
        server.setHandler(handlers);
        return server;
    }

    public static Server publicRest(Config config, MetricRegistry metricRegistry) {
        return buildServer(config, config.getInt("port"), "::", ImmutableMap.of(
                "/unistat", new UnistatServlet(metricRegistry)));
    }

}
