package ru.yandex.infra.sbr_updater;

import java.time.Duration;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

public class Engine {
    private final ScheduledExecutorService executorService;
    private final LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> stageRepo;
    private final SandboxOperations sandboxOperations;
    private final Metrics metrics;
    private final LeaderService leaderService;
    private final Duration updateInterval = Duration.ofDays(1);
    private final Duration retryInterval = Duration.ofMinutes(15);
    private static final Logger LOG = LoggerFactory.getLogger(Engine.class);

    public Engine(ScheduledExecutorService executorService,
                  LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> stageRepo,
                  SandboxOperations sandboxOperations,
                  Metrics metrics,
                  LeaderService leaderService) {
        this.executorService = executorService;
        this.stageRepo = stageRepo;
        this.sandboxOperations = sandboxOperations;
        this.metrics = metrics;
        this.leaderService = leaderService;
    }

    public void start() {
        executorService.submit(this::updateResources);
    }

    private void updateResources() {
        try {
            leaderService.ensureLeadership();
            StageSelector.selectStageResources(stageRepo)
                    .thenApply(allResources -> {
                        metrics.setTotalResources(allResources.size());

                        Set<Resource> resources = allResources.stream()
                                .filter(r -> !r.hasInfiniteTTl())
                                .collect(Collectors.toSet());

                        Set<Long> ids = resources.stream()
                                .map(Resource::getSandboxId)
                                .filter(StringUtils::isNumericArabic)
                                .map(Long::parseLong)
                                .collect(Collectors.toSet());

                        LOG.info("Selected {} resource without inf ttl", resources.size());
                        LOG.info("Detected {} resource ids", ids.size());
                        metrics.setDecayedResources(resources.size());
                        metrics.setResolvedResources(ids.size());

                        return ids;
                    })
                    .thenCompose(sandboxOperations::touchResourcesBatch)
                    .thenAccept(notUpdated -> {
                        LOG.info("Errors during update {} sandbox resources", notUpdated.size());
                        metrics.setSandboxErrors(notUpdated.size());
                    })
                    .orTimeout(1, TimeUnit.MINUTES)
                    .whenComplete((result, error) -> scheduleNextOrFail(updateInterval));
        } catch (Exception e) {
            LOG.error("error during update {}, rescheduling", e.getMessage());
            this.scheduleNextOrFail(retryInterval);
        }
    }

    private void scheduleNextOrFail(Duration delay) {
        try {
            executorService.schedule(this::updateResources, delay.toMillis(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            ExitUtils.gracefulExit(ExitUtils.EXECUTOR_SCHEDULING_FAILURE);
        }
    }
}
