package ru.yandex.infra.sbr_updater;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Throwables;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.eclipse.jetty.server.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.LeaderGaugeRegistry;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

import static ru.yandex.infra.controller.metrics.MetricUtils.buildMetricRegistry;

public class Main {
    private static final Logger LOG = LoggerFactory.getLogger(Main.class);

    public static void main(String[] args) {
        try {
            doMain(args);
        } catch (Exception e) {
            LOG.error("Exception in main:", e);
            ExitUtils.gracefulExit(ExitUtils.EXCEPTION_MAIN);
        }
    }

    public static void doMain(String[] args) {
        LOG.info("Bootstrap");

        Config config = ConfigFactory.load("application_defaults.conf")
                .withFallback(ConfigFactory.systemEnvironmentOverrides());

        MetricRegistry metricRegistry = buildMetricRegistry();
        LeaderService leaderService = ConfigUtils.leaderService(config.getConfig("leader_lock"), metricRegistry);
        GaugeRegistry gaugeRegistry = new LeaderGaugeRegistry(metricRegistry, leaderService);
        GaugeRegistry stageRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "stages");
        Metrics metrics = new Metrics(metricRegistry);

        Server publicRest = ConfigUtils.publicRest(config.getConfig("rest_server"), metricRegistry);
        startServer(publicRest);

        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();

        Config ypConfig = config.getConfig("yp");
        YpRawObjectService ypMasterClient = ConfigUtils.ypRawClient(ypConfig).objectService();

        LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository =
                ConfigUtils.ypStageRepository(ypMasterClient,
                        ypConfig.getInt("request_page_size"),
                        ypConfig.getInt("request_page_size"),
                        ConfigUtils.labels(ypConfig.getConfig("labels")),
                        stageRegistry);

        SandboxApi api = ConfigUtils.getSandboxApi(config.getConfig("sandbox"));
        SandboxOperations sandboxOperations = new SandboxOperations(api, metrics);

        Engine engine = new Engine(executor, ypStageRepository, sandboxOperations, metrics, leaderService);
        engine.start();
        LOG.info("Job started");
    }

    private static void startServer(Server server) {
        try {
            server.start();
        } catch (Exception e) {
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }
}
