package ru.yandex.infra.sbr_updater;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.MoreObjects;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.ListenableFuture;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SandboxApi {

    public static class BatchResults {

        public enum StatusEnum {
            SUCCESS("SUCCESS"),
            WARNING("WARNING"),
            ERROR("ERROR");

            private static final Map<String, StatusEnum> namesMap = new HashMap<>();
            private String value;

            public static BatchResults.StatusEnum forValue(String value) {
                return namesMap.get(value);
            }

            StatusEnum(String value) {
                this.value = value;
            }

            public String toString() {
                return String.valueOf(this.value);
            }

            static {
                namesMap.put("SUCCESS", SUCCESS);
                namesMap.put("WARNING", WARNING);
                namesMap.put("ERROR", ERROR);
            }
        }

        @Override
        public String toString() {
            return MoreObjects.toStringHelper(this)
                    .add("status", status)
                    .add("message", message)
                    .add("id", id)
                    .toString();
        }

        private final BatchResults.StatusEnum status;

        private final String message;

        private final long id;

        public BatchResults(long id, String message, String status) {
            this.id = id;
            this.message = message;
            this.status = BatchResults.StatusEnum.forValue(status);
        }

        public StatusEnum getStatus() {
            return status;
        }

        public String getMessage() {
            return message;
        }

        public long getId() {
            return id;
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(SandboxApi.class);
    private final AsyncHttpClient httpClient;
    private final String baseUrl;
    private final String token;
    private final ObjectMapper mapper = new ObjectMapper();

    public SandboxApi(AsyncHttpClient httpClient, String baseUrl, String token) {
        this.httpClient = httpClient;
        this.baseUrl = baseUrl;
        this.token = token;
    }

    public CompletableFuture<List<BatchResults>> batchResources(Set<Long> objectsList, String operation) {
        try {
            String url = String.format("%s/api/v1.0/batch/resources/%s", baseUrl, operation);

            ObjectNode requestJson = mapper.createObjectNode();
            ArrayNode ids = mapper.createArrayNode();
            objectsList.forEach(ids::add);
            requestJson.set("id", ids);

            return requestPUT(url, mapper.writeValueAsString(requestJson)).toCompletableFuture()
                    .thenApply(response -> {
                        try {
                            JsonNode root = mapper.readTree(response.getResponseBodyAsBytes());
                            ArrayList<BatchResults> result = new ArrayList<>();
                            for (JsonNode item : root) {
                                result.add(new BatchResults(item.get("id").longValue(),
                                        item.get("message").textValue(),
                                        item.get("status").textValue()));
                            }

                            return result;
                        } catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    });
        } catch (Exception e) {
            LOG.error("Sandbox request setup failed", e);
            return CompletableFuture.failedFuture(e);
        }
    }

    private ListenableFuture<Response> requestPUT(String url, String body) {
        return httpClient.executeRequest(httpClient
                .preparePut(url)
                .setBody(body)
                .addHeader("Content-Type", "application/json; charset=utf-8")
                .addHeader("Authorization", String.format("OAuth %s", token))
                .build()
        );
    }
}
