package ru.yandex.infra.sbr_updater;

import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.SelectedObjects;
import ru.yandex.infra.controller.yp.Selector;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.client.pods.TResourceMeta;

import static java.util.Collections.emptyMap;

public class StageSelector {
    private static final Logger LOG = LoggerFactory.getLogger(StageSelector.class);

    private static Set<Resource> selectStaticResources(List<TStageSpec> stages) {
        return stages.stream()
                .flatMap(stage -> stage.getDeployUnitsMap().values().stream())
                .map(du -> du.hasMultiClusterReplicaSet() ?
                        du.getMultiClusterReplicaSet().getReplicaSet().getPodTemplateSpec() :
                        du.getReplicaSet().getReplicaSetTemplate().getPodTemplateSpec())
                .map(pts -> pts.getSpec().getPodAgentPayload().getSpec().getResources())
                .flatMap(rg -> rg.getStaticResourcesList().stream())
                .map(r -> new Resource(r.getUrl(), r.getMeta()))
                .collect(Collectors.toSet());
    }

    public static Set<Resource> selectLayerResources(List<TStageSpec> stages) {
        return stages.stream()
                .flatMap(stage -> stage.getDeployUnitsMap().values().stream())
                .map(du -> du.hasMultiClusterReplicaSet() ?
                        du.getMultiClusterReplicaSet().getReplicaSet().getPodTemplateSpec() :
                        du.getReplicaSet().getReplicaSetTemplate().getPodTemplateSpec())
                .map(pts -> pts.getSpec().getPodAgentPayload().getSpec().getResources())
                .flatMap(rg -> rg.getLayersList().stream())
                .map(l -> new Resource(l.getUrl(), l.getMeta()))
                .collect(Collectors.toSet());
    }

    public static Set<Resource> selectDynamicResources(List<TStageSpec> stages) {
        return stages.stream()
                .flatMap(stage -> stage.getDynamicResourcesMap().values().stream())
                .flatMap(dr -> dr.getDynamicResource().getDeployGroupsList().stream())
                .flatMap(dg -> dg.getUrlsList().stream())
                .map(url -> new Resource(url, TResourceMeta.getDefaultInstance()))
                .collect(Collectors.toSet());
    }

    public static CompletableFuture<List<TStageSpec>> selectStageSpecs(LabelBasedRepository<StageMeta,
            TStageSpec, TStageStatus> ypStageRepository) {
        return ypStageRepository.selectObjects(
                new Selector.Builder()
                        .withSpecAndTimestamps()
                        .build(),
                emptyMap())
                .thenApply(SelectedObjects::getObjects)
                .thenApply(stages -> stages.values().stream()
                        .filter(Try::isSuccess)
                        .map(Try::get)
                        .map(YpObject::getSpec)
                        .collect(Collectors.toList()));
    }

    public static CompletableFuture<Set<Resource>> selectStageResources(LabelBasedRepository<StageMeta, TStageSpec,
            TStageStatus> ypStageRepository) {
        return selectStageSpecs(ypStageRepository).thenApply(stages -> {
            LOG.info("Found stages: {}", stages.size());
            Set<Resource> resources = selectStaticResources(stages);
            Set<Resource> layers = selectLayerResources(stages);
            Set<Resource> dynamicResources = selectDynamicResources(stages);
            LOG.info("Found static resources: {}", resources.size());
            LOG.info("Found layers: {}", layers.size());
            LOG.info("Found dynamic resources: {}", dynamicResources.size());
            resources.addAll(layers);
            resources.addAll(dynamicResources);
            LOG.info("Found total resources: {}", resources.size());
            return resources;
        });
    }
}
