package ru.yandex.infra.sidecars_updater;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.typesafe.config.Config;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;

import ru.yandex.infra.controller.YtSettings;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.ObjectBuilderDescriptor;
import ru.yandex.infra.controller.yp.YpObjectRepository;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxClient;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxClientImpl;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxInfoGetter;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsService;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsServiceProxy;
import ru.yandex.infra.sidecars_updater.sidecar_service.UpdateTaskRunner;
import ru.yandex.infra.sidecars_updater.sidecar_service.YtUpdateTaskRepository;
import ru.yandex.infra.sidecars_updater.sidecars.BoxLayerSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.CoredumpToolsSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.DRUSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.GdbSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.LogbrokerSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.PodAgentSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.RuntimeVersionSidecar;
import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;
import ru.yandex.infra.sidecars_updater.sidecars.TVMSidecar;
import ru.yandex.infra.sidecars_updater.staff.StaffClient;
import ru.yandex.infra.sidecars_updater.staff.StaffClientImpl;
import ru.yandex.infra.sidecars_updater.webauth.WebAuthClientImpl;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.StartrekClientBuilder;
import ru.yandex.yp.YpInstance;
import ru.yandex.yp.YpRawClient;
import ru.yandex.yp.YpRawClientBuilder;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.Autogen;
import ru.yandex.yp.client.api.DataModel.TGroupSpec;
import ru.yandex.yp.client.api.DataModel.TGroupStatus;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static ru.yandex.infra.controller.util.ConfigUtils.rpcOptions;

public class ConfigUtils {
    private static final ObjectBuilderDescriptor<Autogen.TStageMeta, StageMeta> STAGE_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(
                    TStageSpec::newBuilder, TStageStatus::newBuilder, StageMeta::fromProto,
                    Autogen.TStageMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<Autogen.TSchemaMeta, SchemaMeta> GROUP_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(
                    TGroupSpec::newBuilder, TGroupStatus::newBuilder, SchemaMeta::fromProto,
                    Autogen.TSchemaMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<Autogen.TProjectMeta, ProjectMeta> PROJECT_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(
                    TProjectSpec::newBuilder, TProjectStatus::newBuilder, ProjectMeta::fromProto,
                    Autogen.TProjectMeta.getDefaultInstance());
    private static final int DEFAULT_PAGE_SIZE = 1000;

    private static <T> Map<String, T> extractMap(Config config, BiFunction<Config, String, T> extractor) {
        return StreamEx.of(config.root().keySet())
                .toMap(key -> extractor.apply(config, key));
    }

    public static Map<String, String> labels(Config config) {
        return extractMap(config, Config::getString);
    }

    public static YpRawClient ypRawClient(Config ypConfig) {
        Config instanceConfig = ypConfig.getConfig("master");
        Duration mastersTimeout = ypConfig.getDuration("masters_request_timeout");
        Duration requestTimeout = ypConfig.getDuration("request_timeout");

        YpInstance instance = new YpInstance(instanceConfig.getString("host"), instanceConfig.getInt("port"));
        return new YpRawClientBuilder(instance, () -> ypConfig.getString("token"))
                .setTimeout(requestTimeout.toNanos(), TimeUnit.NANOSECONDS)
                .setMastersRequestTimeout(mastersTimeout.toNanos(), TimeUnit.NANOSECONDS)
                .setUseMasterDiscovery(ypConfig.getBoolean("use_master_discovery"))
                .setUsePlaintext(!ypConfig.getBoolean("secure"))
                .setMaxInboundMessageSize(ypConfig.getBytes("inbound_message_size_limit").intValue())
                .setMaxInboundMetadataSize(ypConfig.getBytes("inbound_metadata_size_limit").intValue())
                .build();
    }

    public static Set<String> loadWhiteList(Config config) {
        return ImmutableSet.copyOf(config.getStringList("stage_list.white_list"));
    }

    public static Set<String> loadBlackList(Config config) {
        return ImmutableSet.copyOf(config.getStringList("stage_list.black_list"));
    }

    public static LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository(
            YpRawObjectService objectService, int selectPageSize, int watchPageSize,
            Map<String, String> labels,
            GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.STAGE, labels,
                Optional.empty(),
                objectService, STAGE_DESCRIPTOR, selectPageSize, watchPageSize, gaugeRegistry);
    }

    public static LabelBasedRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository(
            YpRawObjectService objectService, GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.GROUP, Collections.emptyMap(),
                Optional.empty(), objectService, GROUP_DESCRIPTOR, DEFAULT_PAGE_SIZE, gaugeRegistry);
    }

    public static LabelBasedRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectRepository(
            YpRawObjectService objectService, GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.PROJECT, Collections.emptyMap(),
                Optional.empty(), objectService, PROJECT_DESCRIPTOR, DEFAULT_PAGE_SIZE, gaugeRegistry);
    }

    public static StaffClientImpl createStaffClient(Config config) {
        AsyncHttpClient staffHttpClient = httpClient(config.getConfig("http_client"));
        return new StaffClientImpl(staffHttpClient, config.getString("token"));
    }

    public static WebAuthClientImpl createWebAuthClient(Config config) {
        AsyncHttpClient webAuthClient = httpClient(config.getConfig("http_client"));
        var sidecarOwners = EntryStream.of(extractMap(config.getConfig("sidecar_owners"), Config::getStringList))
                .mapKeys(Sidecar.Type::valueOf)
                .toMap();
        return new WebAuthClientImpl(webAuthClient, config.getStringList("whitelist"), sidecarOwners, config.getBoolean("disabled"));
    }

    public static Session createStartrekSession(Config config) {
        return StartrekClientBuilder.newBuilder()
                .uri("https://st-api.yandex-team.ru")
                .connectionTimeout((int) config.getDuration("connection_timeout").toMillis(), MILLISECONDS)
                .socketTimeout((int) config.getDuration("socket_timeout").toMillis(), MILLISECONDS)
                .build(config.getString("token"));
    }

    public static StageUpdateNotifier createNotifier(
            YpObjectRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectRepository,
            YpObjectRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository,
            Session startrekSession, StaffClient staffClient, Config config) {
        return new StageUpdateNotifier(
                ypProjectRepository, ypGroupRepository, startrekSession, staffClient,
                config.getStringList("ignored_logins"), config.getBoolean("add_extra_summonees"),
                config.getBoolean("notify_owners"),
                config.getBoolean("create_release_ticket"),
                config.getString("queue_name")
        );
    }

    public static YtSettings createYtSettings(Config cfg) {
        return new YtSettings(
                cfg.getString("yt_proxy"),
                cfg.getString("token"),
                cfg.getString("user"),
                rpcOptions(cfg.getConfig("rpc_options")));
    }

    public static YtClient createYtClient(YtSettings ytSettings) {
        return YtClient.builder()
                .setCluster(ytSettings.getProxy())
                .setRpcCredentials(new RpcCredentials(ytSettings.getUser(), ytSettings.getToken()))
                .setRpcOptions(ytSettings.getRpcOptions())
                .build();
    }

    public static SidecarsServiceProxy createSidecarsServiceProxy(SidecarsService sidecarsService,
                                                                  StageUpdateNotifier stageUpdateNotifier,
                                                                  UpdateTaskRunner updateTaskRunner,
                                                                  YtUpdateTaskRepository ytUpdateTaskRepository) {
        return new SidecarsServiceProxy(sidecarsService, stageUpdateNotifier, updateTaskRunner, ytUpdateTaskRepository);
    }

    public static UpdateTaskRunner createUpdateTaskRunner(SidecarsService sidecarsService,
                                                          StageUpdateNotifier stageUpdateNotifier,
                                                          YtUpdateTaskRepository ytUpdateTaskRepository,
                                                          Config sidecarUpdateWorkerConfig) {
        return new UpdateTaskRunner(sidecarsService, stageUpdateNotifier, ytUpdateTaskRepository,
                sidecarUpdateWorkerConfig.getInt("attempts_limit"),
                sidecarUpdateWorkerConfig.getDuration("cycle_sleep"));
    }

    public static SandboxClient getSandboxClient(Config config) {
        AsyncHttpClient sandboxHttpClient = httpClient(config.getConfig("http_client"));

        return new SandboxClientImpl(sandboxHttpClient,
                config.getString("base_url"),
                config.getString("token")
        );
    }

    private static Map<Sidecar.Type, Map<String, String>> loadInfraResourceAttributes(Config config) {
        return EntryStream.of(extractMap(config.getConfig("infra_resources"),
                        (resourceConfig, path) -> extractMap(resourceConfig.getConfig(path), Config::getString)))
                .mapKeys(Sidecar.Type::valueOf)
                .toMap();
    }

    public static List<Sidecar> getSidecarList(Config config, SandboxInfoGetter sandboxInfoGetter) {
        Map<Sidecar.Type, Map<String, String>> sidecarsAttributes = loadInfraResourceAttributes(config);
        if (sidecarsAttributes.size() != Sidecar.Type.values().length) {
            throw new RuntimeException("illegal infra_resources config");
        }
        return ImmutableList.of(
                new TVMSidecar(sidecarsAttributes),
                new PodAgentSidecar(sidecarsAttributes),
                new DRUSidecar(sidecarsAttributes),
                new LogbrokerSidecar(sidecarsAttributes),
                new BoxLayerSidecar(
                        sidecarsAttributes,
                        "PortoLayerSearchUbuntuXenialApp",
                        Sidecar.Type.PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP,
                        sandboxInfoGetter
                ),
                new BoxLayerSidecar(
                        sidecarsAttributes,
                        "PortoLayerSearchUbuntuBionicApp",
                        Sidecar.Type.PORTO_LAYER_SEARCH_UBUNTU_BIONIC_APP,
                        sandboxInfoGetter
                ),
                new BoxLayerSidecar(
                        sidecarsAttributes,
                        "PortoLayerSearchUbuntuFocalApp",
                        Sidecar.Type.PORTO_LAYER_SEARCH_UBUNTU_FOCAL_APP,
                        sandboxInfoGetter
                ),
                new CoredumpToolsSidecar(sidecarsAttributes),
                new GdbSidecar(sidecarsAttributes),
                new RuntimeVersionSidecar(sidecarsAttributes)
        );
    }

    private static AsyncHttpClient httpClient(Config config) {
        return new DefaultAsyncHttpClient(new DefaultAsyncHttpClientConfig.Builder()
                .setRequestTimeout((int) config.getDuration("request_timeout").toMillis())
                .setConnectTimeout((int) config.getDuration("connect_timeout").toMillis())
                .build());
    }

}
