package ru.yandex.infra.sidecars_updater;

import java.util.Optional;

import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsService;
import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.TDeployUnitSpec;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

public class DeployUnitSidecarInfo {
    private final Sidecar.Type sidecarType;
    private final StageMeta stageMeta;
    private final boolean enableAutoUpdate;
    private final String deployUnitName;
    private final long latestSidecarRev;
    private final Optional<Long> currentRev;
    private final Optional<Long> targetRev;
    private final int patchersRevision;
    private final Optional<Integer> targetPatchersRevision;
    private final boolean isUsed;

    public DeployUnitSidecarInfo(YpObject<StageMeta, TStageSpec, TStageStatus> stage, String deployUnitName,
                                 Sidecar sidecar) {
        Optional<TDeployUnitSpec> currentDU =
                Optional.ofNullable(stage.getSpec().getDeployUnitsMap().get(deployUnitName));
        this.currentRev = currentDU
                .flatMap(sidecar.getRevisionGetter())
                .filter(rev -> rev > 0);
        this.targetRev = Optional.ofNullable(stage.getLabels().get(SidecarsService.DU_SIDECAR_TARGET_LABEL))
                .filter(YTreeNode::isMapNode)
                .flatMap(n -> n.mapNode().get(deployUnitName))
                .flatMap(n -> n.mapNode().get(sidecar.getLabelName()))
                .map(YTreeNode::longValue);
        this.patchersRevision = currentDU
                .stream()
                .mapToInt(TDeployUnitSpec::getPatchersRevision)
                .findFirst().orElse(0);
        this.targetPatchersRevision = Optional.ofNullable(stage.getLabels().get(SidecarsService.DU_PATCHERS_TARGET_LABEL))
                .filter(YTreeNode::isMapNode)
                .flatMap(n -> n.mapNode().get(deployUnitName))
                .map(YTreeNode::intValue);
        sidecarType = sidecar.getResourceType();
        latestSidecarRev = sidecar.getRevision();
        this.deployUnitName = deployUnitName;
        this.enableAutoUpdate = currentDU
                .map(du -> du.getInfraComponents().getAllowAutomaticUpdates())
                .orElse(false);
        this.stageMeta = stage.getMeta();
        this.isUsed = sidecar.isUsedBy(stage.getSpec(), deployUnitName);
    }

    public Optional<Long> getCurrentRev() {
        return currentRev;
    }

    public Optional<Long> getTargetRev() {
        return targetRev;
    }

    public int getPatchersRevision() {
        return patchersRevision;
    }

    public Optional<Integer> getTargetPatchersRevision() {
        return targetPatchersRevision;
    }

    public long getLatestSidecarRev() {
        return latestSidecarRev;
    }

    public Sidecar.Type getSidecarType() {
        return sidecarType;
    }

    public String getStageName() {
        return stageMeta.getId();
    }

    public String getDeployUnitName() {
        return deployUnitName;
    }

    public boolean isEnableAutoUpdate() {
        return enableAutoUpdate;
    }

    public StageMeta getStageMeta() {
        return stageMeta;
    }

    public boolean isUsed() {
        return isUsed;
    }

    public String getFullDeployUnitName() {
        return stageMeta.getId() + "." + deployUnitName;
    }
}
