package ru.yandex.infra.sidecars_updater;

import java.time.Duration;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.sidecars_updater.metrics.SidecarMetricsService;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsService;


public class Engine {
    private final ScheduledExecutorService executorService;
    private final SidecarsService sidecarsService;
    private final SidecarMetricsService sidecarMetricsService;
    private final StatisticsService statisticsService;
    private final StageCache stageCache;
    private final Duration cacheUpdateInterval;
    private final boolean isSendMetricToYt;
    private final Duration ytUpdateInterval;
    private static final Logger LOG = LoggerFactory.getLogger(Engine.class);


    public Engine(ScheduledExecutorService executorService,
                  SidecarsService sidecarsService,
                  SidecarMetricsService sidecarMetricsService,
                  StatisticsService statisticsService,
                  StageCache stageCache,
                  Duration cacheUpdateInterval,
                  boolean isSendMetricToYt,
                  Duration ytUpdateInterval) {
        this.executorService = executorService;
        this.sidecarsService = sidecarsService;
        this.sidecarMetricsService = sidecarMetricsService;
        this.statisticsService = statisticsService;
        this.stageCache = stageCache;
        this.cacheUpdateInterval = cacheUpdateInterval;
        this.isSendMetricToYt = isSendMetricToYt;
        this.ytUpdateInterval = ytUpdateInterval;
    }

    public void start() {
        executorService.submit(this::collectDeployUnitsInfo);
        if (isSendMetricToYt) {
            executorService.submit(this::updateYtStatisticsTables);
        }
    }

    private void collectDeployUnitsInfo() {
        sidecarsService.refreshData();
        LOG.info("deploy units info collecting...");
        sidecarsService.collectDeployUnitData().thenAccept(infos -> sidecarsService.getSidecars()
                .forEach(sidecar -> sidecarMetricsService.doUpdate(infos, sidecar.getResourceType()))
        );
        stageCache.getStagesFutureCache().thenAccept(statisticsService::updateStatistics);
        scheduleOrFail(this::collectDeployUnitsInfo, cacheUpdateInterval);
    }

    private void updateYtStatisticsTables() {
        statisticsService.updateYt();
        LOG.info("YT tables was updated");
        scheduleOrFail(this::updateYtStatisticsTables, ytUpdateInterval);
    }

    private void scheduleOrFail(Runnable method, Duration updateInterval) {
        try {
            executorService.schedule(method, updateInterval.toMillis(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            LOG.error(e.getMessage(), e);
            ExitUtils.gracefulExit(ExitUtils.EXECUTOR_SCHEDULING_FAILURE);
        }
    }
}
