package ru.yandex.infra.sidecars_updater;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.util.YsonUtils;
import ru.yandex.infra.controller.yp.Paths;
import ru.yandex.infra.controller.yp.UpdateYpObjectRequest;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeNodeUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpObjectUpdate;
import ru.yandex.yp.model.YpSetUpdate;
import ru.yandex.yp.model.YpTypedId;

public class LabelUpdater<Spec extends Message, Status extends Message> {
    private static final Logger LOG = LoggerFactory.getLogger(LabelUpdater.class);

    private final YpTransactionClient transactionClient;
    private final YpRawObjectService ypClient;
    private final Map<String, Map<String, YTreeNode>> stageLabelsCache = new ConcurrentHashMap<>();

    public LabelUpdater(YpRawObjectService ypClient,
                        YpTransactionClient transactionClient) {
        this.transactionClient = transactionClient;
        this.ypClient = ypClient;
    }

    public void refreshLabelsInfo(String name, Map<String, YTreeNode> labels) {
        stageLabelsCache.put(name, labels);
    }

    public CompletableFuture<?> patchLabels(String stageId, Map<String, YTreeNode> patches) {
        stageLabelsCache.putIfAbsent(stageId, Collections.emptyMap());

        final Map<String, YTreeNode> update = new HashMap<>();
        for (Map.Entry<String, YTreeNode> entry : patches.entrySet()) {
            final String labelName = entry.getKey();
            final YTreeNode patch = entry.getValue();

            YTreeNode currentLabel = stageLabelsCache
                    .getOrDefault(stageId, Collections.emptyMap())
                    .getOrDefault(labelName, patch);

            YTreeBuilder builder = YTree.builder();
            YTreeNodeUtils.merge(currentLabel, patch, builder, true);
            YTreeNode newLabels = builder.build();
            update.put(labelName, newLabels);
        }

        UpdateYpObjectRequest<Spec, Status> updateRequest =
                new UpdateYpObjectRequest.Builder<Spec, Status>()
                        .setLabels(update)
                        .build();

        return transactionClient.runWithTransaction(transaction -> {
            YpObjectUpdate.Builder builder = YpObjectUpdate.builder(new YpTypedId(stageId, YpObjectType.STAGE));
            builder.addAttributeTimestampPrerequisites(updateRequest.getPrerequisites());
            updateRequest.getLabels().forEach((name, node) ->
                    builder.addSetUpdate(new YpSetUpdate(Paths.LABELS + Paths.DELIMITER + name, node, YsonUtils::toYsonPayload)));

            return ypClient.updateObject(builder.build(), transaction);
        }).thenApply(res -> {
            stageLabelsCache.get(stageId).putAll(update);
            return res;
        });
    }
}
