package ru.yandex.infra.sidecars_updater;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;

import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import org.eclipse.jetty.server.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.YtSettings;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.MetricAdapterRegistry;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.servlets.UnistatServlet;
import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.Selector;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.infra.controller.yp.YpTransactionClientImpl;
import ru.yandex.infra.sidecars_updater.metrics.Metrics;
import ru.yandex.infra.sidecars_updater.metrics.SidecarMetricsService;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxClient;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxInfoGetter;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxInfoGetterImpl;
import ru.yandex.infra.sidecars_updater.servlets.ApiServlet;
import ru.yandex.infra.sidecars_updater.servlets.GroupStatServlet;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsService;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsServiceProxy;
import ru.yandex.infra.sidecars_updater.sidecar_service.UpdateTaskRunner;
import ru.yandex.infra.sidecars_updater.sidecar_service.YtUpdateTaskRepository;
import ru.yandex.infra.sidecars_updater.staff.StaffClientImpl;
import ru.yandex.infra.sidecars_updater.statistics.StatisticsRepository;
import ru.yandex.infra.sidecars_updater.webauth.WebAuthClient;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.startrek.client.Session;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yt.ytclient.proxy.YtClient;

import static ru.yandex.infra.controller.metrics.MetricUtils.buildMetricRegistry;
import static ru.yandex.infra.controller.servlets.ServerBuilder.buildServer;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createNotifier;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createSidecarsServiceProxy;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createStaffClient;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createStartrekSession;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createUpdateTaskRunner;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createWebAuthClient;
import static ru.yandex.infra.sidecars_updater.ConfigUtils.createYtClient;
import static ru.yandex.infra.sidecars_updater.servlets.ApiServlet.Operation.APPLY;
import static ru.yandex.infra.sidecars_updater.servlets.ApiServlet.Operation.APPLY_SIDECAR;
import static ru.yandex.infra.sidecars_updater.servlets.ApiServlet.Operation.LIST_SIDECAR_RELEASES;
import static ru.yandex.infra.sidecars_updater.servlets.ApiServlet.Operation.PATCHER_STATISTIC;
import static ru.yandex.infra.sidecars_updater.servlets.ApiServlet.Operation.SIDECAR_STATISTIC;
import static ru.yandex.infra.sidecars_updater.servlets.ApiServlet.Operation.SIDECAR_UPDATE;

public class Main {
    private static final Logger LOG = LoggerFactory.getLogger(Main.class);

    public static void main(String[] args) {
        try {
            doMain(args);
        } catch (Exception e) {
            LOG.error("Exception in main:", e);
            ExitUtils.gracefulExit(ExitUtils.EXCEPTION_MAIN);
        }
    }

    public static void doMain(String[] args) {
        LOG.info("Bootstrap");

        Config config = ConfigFactory.parseFile(new File(args[0]), ConfigParseOptions.defaults().setAllowMissing(false))
                .withFallback(ConfigFactory.load("application_defaults.conf"))
                .withFallback(ConfigFactory.systemEnvironmentOverrides());
        Config mainConfig = config.getConfig("main");

        MetricRegistry metricRegistry = buildMetricRegistry();
        GaugeRegistry gaugeRegistry = new MetricAdapterRegistry(metricRegistry);
        GaugeRegistry stageRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "stages");
        Metrics metrics = new Metrics(metricRegistry);

        Config ypConfig = config.getConfig("yp");
        YpRawObjectService ypMasterClient = ConfigUtils.ypRawClient(ypConfig).objectService();

        LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository =
                ConfigUtils.ypStageRepository(ypMasterClient,
                        ypConfig.getInt("request_page_size"),
                        ypConfig.getInt("request_page_size"),
                        ConfigUtils.labels(ypConfig.getConfig("labels")),
                        stageRegistry);

        StageCache stageCache = new StageCache(ypStageRepository);

        LabelBasedRepository<SchemaMeta, DataModel.TGroupSpec, DataModel.TGroupStatus> ypGroupRepository =
                ConfigUtils.ypGroupRepository(ypMasterClient,
                        new NamespacedGaugeRegistry(gaugeRegistry, "groups"));
        LabelBasedRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectRepository =
                ConfigUtils.ypProjectRepository(ypMasterClient,
                        new NamespacedGaugeRegistry(gaugeRegistry, "projects"));
        StaffClientImpl staffClient = createStaffClient(config.getConfig("staff"));
        Session startrekSession = createStartrekSession(config.getConfig("startrek"));
        WebAuthClient webAuthClient = createWebAuthClient(config.getConfig("webauth"));

        StageUpdateNotifier stageUpdateNotifier = createNotifier(
                ypProjectRepository, ypGroupRepository, startrekSession, staffClient, mainConfig.getConfig("notifier"));

        if (args.length == 2) {
            PrintResponsibleForStages(args[1], ypStageRepository, stageUpdateNotifier);
            System.exit(0);
        }

        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();

        Config sandboxConfig = config.getConfig("sandbox");
        SandboxClient sandboxClient = ConfigUtils.getSandboxClient(sandboxConfig);

        YpTransactionClient ypTransactionMasterClient = new YpTransactionClientImpl(ypMasterClient);
        LabelUpdater<TStageSpec, TStageStatus> labelUpdater = new LabelUpdater<>(ypMasterClient,
                ypTransactionMasterClient);

        Set<String> blackList = ConfigUtils.loadBlackList(mainConfig);
        Set<String> whiteList = ConfigUtils.loadWhiteList(mainConfig);
        SandboxInfoGetter sandboxInfoGetter = new SandboxInfoGetterImpl(
                sandboxClient,
                sandboxConfig.getDuration("request_wait_timeout"),
                sandboxConfig.getLong("resources_size_limit")
        );

        SidecarsService sidecarsService = new SidecarsService(
                labelUpdater,
                ConfigUtils.getSidecarList(config, sandboxInfoGetter),
                sandboxClient,
                sandboxInfoGetter,
                stageUpdateNotifier,
                blackList,
                whiteList,
                stageCache
        );
        Config ytUpdateWorkerConfig = config.getConfig("yt").getConfig("updater");
        YtSettings ytSettings = ConfigUtils.createYtSettings(ytUpdateWorkerConfig);
        YtClient client = createYtClient(ytSettings);
        YtUpdateTaskRepository ytUpdateTaskRepository = new YtUpdateTaskRepository(client,
                ytUpdateWorkerConfig.getString("tables_folder_path"));

        Config sidecarUpdateWorkerConfig = mainConfig.getConfig("sidecar_update_worker");
        UpdateTaskRunner updateTaskRunner = createUpdateTaskRunner(sidecarsService, stageUpdateNotifier,
                ytUpdateTaskRepository, sidecarUpdateWorkerConfig);
        SidecarsServiceProxy sidecarsServiceProxy = createSidecarsServiceProxy(sidecarsService, stageUpdateNotifier,
                updateTaskRunner, ytUpdateTaskRepository);
        updateTaskRunner.run();

        Config ytStatConfig = config.getConfig("yt").getConfig("stat");
        Yt ytStat = YtUtils.http(ytStatConfig.getString("yt_proxy"), ytStatConfig.getString("token"));
        String tablesFolderPathStat = ytStatConfig.getString("tables_folder_path");

        createAndStartHttpServers(config.getConfig("http_server"), sidecarsService, sidecarsServiceProxy,
                metricRegistry, webAuthClient);

        LabelStatisticsUpdater labelStatisticsUpdater = new LabelStatisticsUpdater();
        StatisticsService statisticsService = new StatisticsService(metricRegistry, ytStat.tables(),
                tablesFolderPathStat,
                labelStatisticsUpdater);
        statisticsService.registerStatistic(StatisticsRepository.getBasicStatistics());
        Duration cacheUpdateInterval = mainConfig.getDuration("cache_update_interval");
        boolean isSendMetricToYt = ytStatConfig.getBoolean("send_metrics");
        Duration ytUpdateInterval = ytStatConfig.getDuration("update_interval");

        SidecarMetricsService sidecarMetricsService = new SidecarMetricsService(metrics);
        Engine engine = new Engine(executor, sidecarsService, sidecarMetricsService, statisticsService, stageCache,
                cacheUpdateInterval, isSendMetricToYt, ytUpdateInterval);
        engine.start();
        LOG.info("Job started");

    }

    private static void createAndStartHttpServers(Config config,
                                                  SidecarsService sidecarsService,
                                                  SidecarsServiceProxy sidecarsServiceProxy,
                                                  MetricRegistry metricRegistry,
                                                  WebAuthClient webAuthClient) {
        List<Server> servers = getServers(config, metricRegistry, sidecarsService, sidecarsServiceProxy, webAuthClient);

        try {
            for (Server server : servers) {
                server.start();
            }
        } catch (Exception e) {
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }

    private static List<Server> getServers(Config config,
                                           MetricRegistry metricRegistry,
                                           SidecarsService sidecarsService,
                                           SidecarsServiceProxy sidecarsServiceProxy,
                                           WebAuthClient webAuthClient) {
        Map<String, Map<String, HttpServlet>> servlets = ImmutableMap.of(
                "private_api", Map.of(
                        "list-sidecars", new ApiServlet(sidecarsService, sidecarsServiceProxy, LIST_SIDECAR_RELEASES,
                                webAuthClient),
                        "stat-sidecars", new ApiServlet(sidecarsService, sidecarsServiceProxy, SIDECAR_STATISTIC,
                                webAuthClient),
                        "stat-patchers", new ApiServlet(sidecarsService, sidecarsServiceProxy, PATCHER_STATISTIC,
                                webAuthClient),
                        "groupstat", new GroupStatServlet(),
                        // FIXME: deprecated
                        "apply-sidecar", new ApiServlet(sidecarsService, sidecarsServiceProxy, APPLY_SIDECAR,
                                webAuthClient),
                        "apply", new ApiServlet(sidecarsService, sidecarsServiceProxy, APPLY, webAuthClient),
                        "update-status", new ApiServlet(sidecarsService, sidecarsServiceProxy, SIDECAR_UPDATE,
                                webAuthClient)
                ),
                "unistat", Map.of(
                        "unistat", new UnistatServlet(metricRegistry)
                )
        );

        return config.root().keySet().stream()
                .map(entry -> buildServer(config.getConfig(entry), servlets.get(entry)))
                .collect(Collectors.toList());
    }

    private static void PrintResponsibleForStages(
            String stagesFilePath,
            LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository,
            StageUpdateNotifier stageUpdateNotifier) {
        List<String> stages = new ArrayList<>();

        try (BufferedReader br = new BufferedReader(new FileReader(stagesFilePath))) {
            while (true) {
                String stageId = br.readLine();
                if (stageId == null) {
                    break;
                }
                stages.add(stageId);
            }
        } catch (IOException e) {
            LOG.error("Exception in main:", e);
        }

        for (String id : stages) {
            try {
                YpObject<StageMeta, TStageSpec, TStageStatus> stage =
                        ypStageRepository.getObject(id, new Selector.Builder().withMeta().build()).get().get();
                System.out.printf("Stage %s: %s%n", id, stageUpdateNotifier.getResponsible(stage.getMeta()));
            } catch (Exception e) {
                System.err.printf("Stage %s: %s%n", id, e);
            }
        }
    }
}
