package ru.yandex.infra.sidecars_updater;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.function.Function;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.Selector;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.controller.yp.YpObjectRepository;
import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;
import ru.yandex.infra.sidecars_updater.staff.StaffClient;
import ru.yandex.infra.sidecars_updater.staff.StaffClientError;
import ru.yandex.startrek.client.Session;
import ru.yandex.startrek.client.model.CommentCreate;
import ru.yandex.startrek.client.model.Issue;
import ru.yandex.startrek.client.model.IssueCreate;
import ru.yandex.startrek.client.model.IssueUpdate;
import ru.yandex.yp.client.api.AccessControl;
import ru.yandex.yp.client.api.DataModel.TGroupSpec;
import ru.yandex.yp.client.api.DataModel.TGroupStatus;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;

import static java.lang.String.format;
import static java.util.Collections.emptyList;

public class StageUpdateNotifier {
    private static final Logger LOG = LoggerFactory.getLogger(StageUpdateNotifier.class);
    private static final Selector SPEC_SELECTOR = new Selector.Builder().withSpecAndTimestamps().build();
    private static final Selector META_SELECTOR = new Selector.Builder().withMeta().build();
    private static final String IDM_GROUP_PREFIX = "idm:";
    private static final String ABC_GROUP_PREFIX = "abc:service:";
    private static final List<String> GROUP_PREFIXES = List.of(IDM_GROUP_PREFIX, ABC_GROUP_PREFIX);
    private static final int EXTRA_SUMMONEES_COUNT = 2;
    private static final String ROBOT_NOTIFIER = "robot-deploy-info";

    private final YpObjectRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectRepository;
    private final YpObjectRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository;
    private final Session startrekSession;
    private final StaffClient staffClient;
    private final List<String> ignoredLogins;
    private final boolean addExtraSummonees;
    private final boolean notifyOwners;
    private final boolean createReleaseTicket;
    private final String startrekQueueName;

    public StageUpdateNotifier(YpObjectRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectRepository,
                               YpObjectRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository,
                               Session startrekSession, StaffClient staffClient, List<String> ignoredLogins,
                               boolean addExtraSummonees, boolean notifyOwners, boolean createReleaseTicket,
                               String startrekQueueName) {
        this.ypProjectRepository = ypProjectRepository;
        this.ypGroupRepository = ypGroupRepository;
        this.startrekSession = startrekSession;
        this.staffClient = staffClient;
        this.ignoredLogins = ignoredLogins;
        this.addExtraSummonees = addExtraSummonees;
        this.notifyOwners = notifyOwners;
        this.createReleaseTicket = createReleaseTicket;
        this.startrekQueueName = startrekQueueName;
    }

    public void updateNotify(StageMeta stageMeta, List<Sidecar> sidecars) throws StageUpdateNotifierError {
        String stageId = stageMeta.getId();
        String projectId = stageMeta.getId();

        List<String> responsible = tryToExtractResponsible(emptyList(), this::getStageAclResponsible, stageMeta,
                result -> notify(result, stageId, sidecars, format("RESPONSIBLE role of stage %s", stageId)));

        ProjectMeta projectMeta = getProjectMeta(stageMeta.getProjectId());
        String projectCreator = projectMeta.getOwnerId();

        responsible = tryToExtractResponsible(responsible, this::getProjectAclResponsible, projectMeta,
                result -> notify(result, stageId, sidecars, format("RESPONSIBLE role of project %s", projectId)));
        responsible = tryToExtractResponsible(responsible, this::getStageAclMaintainer, stageMeta,
                result -> notify(result, stageId, sidecars, format("MAINTAINER role of stage %s", stageId)));

        if (responsible.isEmpty() && isValidUser(projectCreator)) {
            notify(List.of(projectCreator), stageId, sidecars, format("owner_id of project %s", projectId));
            return;
        }

        responsible = tryToExtractResponsible(responsible, this::getProjectAclOwner, projectMeta,
                result -> notify(result, stageId, sidecars, format("OWNER role of project %s", projectId)));
        responsible = tryToExtractResponsible(responsible, this::getProjectAclMaintainer, projectMeta,
                result -> notify(result, stageId, sidecars, format("MAINTAINER role of project %s", projectId)));

        if (responsible.isEmpty()) {
            throw new StageUpdateNotifierError(
                    format("Owner was not found for stage %s, project.owner_id=%s", stageId, projectCreator));
        }
    }

    public Optional<Issue> getIssue(String id) {
        return startrekSession.issues().getO(id).toOptional();
    }

    public Optional<Issue> crateUpdateNotifyMultipleStages(Optional<Sidecar> sidecar, OptionalInt patcher,
                                                           int percent, String initiator) {
        if (!createReleaseTicket) {
            return Optional.empty();
        }

        try {
            IssueCreate issueCreate = getIssueCreateMultipleStages(sidecar, patcher, percent, initiator);
            Issue issue = startrekSession.issues().create(issueCreate, false, true);
            LOG.info("Issue {} was successfully created", issue.getKey());
            return Optional.of(issue);
        } catch (Exception e) {
            LOG.error("Notification exception: {} ", e.getMessage());
        }
        return Optional.empty();
    }

    public void updateIssue(Issue issue, String text) {
        var updateBuilder = IssueUpdate.builder();
        if (issue.getDescription().isPresent()) {
            updateBuilder.description(issue.getDescription().get() + "\n" + text);
        } else {
            updateBuilder.description(text);
        }
        issue.update(updateBuilder.build());
    }

    public void commentIssue(Issue issue, String text) {
        var updateBuilder = IssueUpdate.builder();
        updateBuilder.comment(CommentCreate.comment(text).build());
        issue.update(updateBuilder.build());

    }

    public List<String> getResponsible(StageMeta stageMeta) throws StageUpdateNotifierError {
        List<String> responsible = tryToExtractResponsible(emptyList(), this::getStageAclResponsible, stageMeta);

        ProjectMeta projectMeta = getProjectMeta(stageMeta.getProjectId());
        String projectCreator = projectMeta.getOwnerId();

        responsible = tryToExtractResponsible(responsible, this::getProjectAclResponsible, projectMeta);
        responsible = tryToExtractResponsible(responsible, this::getStageAclMaintainer, stageMeta);

        if (responsible.isEmpty() && isValidUser(projectCreator)) {
            return List.of(projectCreator);
        }

        responsible = tryToExtractResponsible(responsible, this::getProjectAclOwner, projectMeta);
        responsible = tryToExtractResponsible(responsible, this::getProjectAclMaintainer, projectMeta);

        return responsible;
    }

    private <Meta> List<String> tryToExtractResponsible(List<String> currentResponsible,
                                                        Function<Meta, List<String>> responsibleGetter,
                                                        Meta meta) {
        return tryToExtractResponsible(currentResponsible, responsibleGetter, meta, (responsible) -> {
        });
    }

    private <Meta> List<String> tryToExtractResponsible(List<String> currentResponsible,
                                                        Function<Meta, List<String>> responsibleGetter,
                                                        Meta meta,
                                                        Consumer<List<String>> onSuccess) {
        if (currentResponsible.isEmpty()) {
            List<String> responsible = responsibleGetter.apply(meta);
            if (!responsible.isEmpty()) {
                onSuccess.accept(responsible);
            }
            return responsible;
        }

        return currentResponsible;
    }

    private List<String> getStageAclMaintainer(StageMeta meta) {
        String subjectPattern = format("^deploy(-test|-pre)?:%s\\.%s\\.MAINTAINER", meta.getProjectId(), meta.getId());
        return getMemberForSubjectPattern(meta, subjectPattern);
    }

    private List<String> getProjectAclOwner(ProjectMeta meta) {
        String subjectPattern = format("^deploy(-test|-pre)?:%s\\.OWNER$", meta.getId());
        return getMemberForSubjectPattern(meta, subjectPattern);
    }

    private List<String> getProjectAclMaintainer(ProjectMeta meta) {
        String subjectPattern = format("^deploy(-test|-pre)?:%s\\.MAINTAINER", meta.getId());
        return getMemberForSubjectPattern(meta, subjectPattern);
    }

    private List<String> getProjectAclResponsible(ProjectMeta meta) {
        String subjectPattern = format("^deploy(-test|-pre)?:%s\\.RESPONSIBLE", meta.getId());
        return getMemberForSubjectPattern(meta, subjectPattern);
    }

    private List<String> getStageAclResponsible(StageMeta meta) {
        String subjectPattern = format("^deploy(-test|-pre)?:%s\\.%s\\.RESPONSIBLE", meta.getProjectId(), meta.getId());
        return getMemberForSubjectPattern(meta, subjectPattern);
    }

    private List<String> extractFewOwners(List<String> allOwners) {
        return addExtraSummonees
                ? List.copyOf(allOwners.subList(0, Math.min(EXTRA_SUMMONEES_COUNT + 1, allOwners.size())))
                : List.of(allOwners.get(0));
    }

    private List<String> getMemberForSubjectPattern(SchemaMeta meta, String pattern) {
        for (AccessControl.TAccessControlEntry ace : meta.getAcl().getEntries()) {
            if (ace.getSubjectsCount() == 1 && Pattern.matches(pattern, ace.getSubjects(0))) {
                String subject = ace.getSubjects(0);

                TGroupSpec groupSpec = getGroupSpec(subject, meta.getId());
                List<String> owners = groupSpec.getMembersList().stream()
                        .filter(this::isValidUser)
                        .collect(Collectors.toList());
                if (!owners.isEmpty()) {
                    return extractFewOwners(owners);
                }

                for (String groupPrefix : GROUP_PREFIXES) {
                    owners = groupSpec.getMembersList().stream()
                            .filter(member -> member.startsWith(groupPrefix))
                            .map(group -> getAllUsersOfGroup(group, meta.getId()))
                            .flatMap(Collection::stream)
                            .collect(Collectors.toList());
                    if (!owners.isEmpty()) {
                        return extractFewOwners(owners);
                    }
                }
            }
        }
        return emptyList();
    }

    private List<String> getAllUsersOfGroup(String groupId, String stageId) {
        List<String> members = new ArrayList<>();
        TGroupSpec groupSpec = getGroupSpec(groupId, stageId);
        groupSpec.getMembersList().stream().filter(this::isValidUser).forEach(members::add);
        groupSpec.getMembersList().stream()
                .filter(member -> member.startsWith(IDM_GROUP_PREFIX) || member.startsWith(ABC_GROUP_PREFIX))
                .map(idmGroup -> getAllUsersOfGroup(idmGroup, stageId))
                .forEach(members::addAll);
        return members;
    }

    private TGroupSpec getGroupSpec(String groupId, String stageId) {
        try {
            Optional<YpObject<SchemaMeta, TGroupSpec, TGroupStatus>> groupOpt =
                    ypGroupRepository.getObject(groupId, SPEC_SELECTOR).get();
            if (groupOpt.isEmpty()) {
                LOG.error(format("Error: group %s from subjects of stage %s does not exist", groupId, stageId));
                return TGroupSpec.getDefaultInstance();
            }
            return groupOpt.get().getSpec();
        } catch (InterruptedException | ExecutionException e) {
            LOG.error("Error while sending request in YP:", e);
            return TGroupSpec.getDefaultInstance();
        }
    }

    private boolean isValidUser(String member) {
        try {
            return !member.startsWith(IDM_GROUP_PREFIX) &&
                    !member.startsWith(ABC_GROUP_PREFIX) &&
                    !member.startsWith("robot-") &&
                    !member.startsWith("zomb-") &&
                    !member.equals("root") &&
                    !ignoredLogins.contains(member) &&
                    !staffClient.isPersonDismissed(member);
        } catch (StaffClientError staffClientError) {
            LOG.error("Error while sending request to staff. ", staffClientError);
            return false;
        }
    }

    private ProjectMeta getProjectMeta(String projectId) throws StageUpdateNotifierError {
        try {
            Optional<YpObject<ProjectMeta, TProjectSpec, TProjectStatus>> project =
                    ypProjectRepository.getObject(projectId, META_SELECTOR).get();
            if (project.isEmpty()) {
                throw new StageUpdateNotifierError(format("Project %s does not exist", projectId));
            }
            return project.get().getMeta();
        } catch (InterruptedException | ExecutionException e) {
            throw new StageUpdateNotifierError(format("Error while getting project %s", projectId));
        }
    }

    private static IssueCreate getIssueCreate(List<String> owners, String stageId, List<Sidecar> sidecars) {
        StringBuilder sidecarsDescription = new StringBuilder();
        for (Sidecar sidecar : sidecars) {
            sidecarsDescription.append(format("- %s,\n", sidecar));
        }
        return IssueCreate.builder()
                .queue("DEPLOYRELEASES")
                .summary(format("Обновление сайдкаров Деплоя у стейджа %s", stageId))
                .description(format(
                        "Инфраструктурные сайдкары стейджа ((https://deploy.yandex-team.ru/stage/%s %s)):\n\n"
                                + sidecarsDescription
                                + "\nбудут автоматически обновлены со следующей выкладкой.\n\n"
                                + "Если вы не готовы в следующей выкладке обновлять инфраструктурные сайдкары, " +
                                "сообщите нам об этом.",
                        stageId, stageId))
                .author(ROBOT_NOTIFIER)
                .type(2)
                .assignee(owners.get(0))
                .comment(CommentCreate.builder().summonees(Cf.toList(owners)).build())
                .build();
    }

    private IssueCreate getIssueCreateMultipleStages(Optional<Sidecar> sidecar,
                                                     OptionalInt patcher,
                                                     int percent,
                                                     String initiator) {
        String summary = format("Обновление %s %s на %d %% by %s",
                sidecar.isPresent() ? format("сайдкара %s на версию %d", sidecar.get().getLabelName(),
                        sidecar.get().getRevision()) : "",
                patcher.isPresent() ? format("патчера на версию %d", patcher.getAsInt()) : "",
                percent, initiator);
        return IssueCreate.builder()
                .queue(startrekQueueName)
                .summary(summary)
                .author(ROBOT_NOTIFIER)
                .description("")
                .build();
    }

    private void notify(List<String> owners, String stageId, List<Sidecar> sidecars, String responsibleDescription) {
        LOG.info("Responsible {} for stage {} was found in {}.", owners, stageId, responsibleDescription);
        if (!notifyOwners) {
            return;
        }

        IssueCreate issueCreate = getIssueCreate(owners, stageId, sidecars);
        Issue issue = startrekSession.issues().create(issueCreate, false, true);
        LOG.info("Issue {} for users {} about stage {} was successfully created",
                issue.getKey(),
                owners,
                stageId);
    }
}
