package ru.yandex.infra.sidecars_updater;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.codahale.metrics.MetricRegistry;

import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.sidecars_updater.statistics.Statistics;
import ru.yandex.infra.sidecars_updater.statistics.StatisticsRepository;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.tables.YtTables;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

public class StatisticsService {
    private static final String unistat = "unistat";
    private static final String date = "Date";
    private static final String statisticsName = "Statistics_name";
    private static final String statisticsValue = "Statistics_value";
    private static final String duStageStatistics = "du_stage_statistics";
    private static final String duOrStageName = "Du_or_stage_name";
    private final MetricRegistry metricRegistry;
    private final YtTables ytTables;
    private final String tableFolderPath;
    private final LabelStatisticsUpdater labelStatisticsUpdater;
    private final List<Statistics> statistics;

    public StatisticsService(MetricRegistry metricRegistry,
                             YtTables ytTables,
                             String tablesFolderPath,
                             LabelStatisticsUpdater labelStatisticsUpdater) {
        this.metricRegistry = metricRegistry;
        this.ytTables = ytTables;
        this.tableFolderPath = tablesFolderPath;
        this.labelStatisticsUpdater = labelStatisticsUpdater;
        this.statistics = new ArrayList<>();
    }

    public void registerStatistic(List<? extends Statistics> statistics) {
        this.statistics.addAll(statistics);
        statistics.forEach(statistic -> statistic.prepare(metricRegistry));
    }

    public void updateStatistics(Map<String, YpObject<StageMeta, TStageSpec, TStageStatus>> stages) {
        registerStatistic(labelStatisticsUpdater.getNewLabelStatistics(stages));
        statistics.forEach(statistic -> statistic.update(metricRegistry, stages));
    }

    public void updateYt() {
        long daysInUnixTime = System.currentTimeMillis() / (1000 * 60 * 60 * 24);
        writeToYtTable(
                tableFolderPath + "/" + unistat,
                true,
                StatisticsRepository.statisticsResults.entrySet().stream().map(entry ->
                        YTree.mapBuilder()
                                .key(date).value(daysInUnixTime)
                                .key(statisticsName).value(entry.getKey())
                                .key(statisticsValue).value(entry.getValue())
                                .buildMap()
                ).collect(Collectors.toList())
        );
        writeToYtTable(
                tableFolderPath + "/" + duStageStatistics,
                false,
                StatisticsRepository.statisticsGroupResults.entrySet().stream()
                        .flatMap(statNameToGroupsByValues -> statNameToGroupsByValues.getValue().entrySet().stream()
                                .flatMap(statValueToDuOrStageName -> statValueToDuOrStageName.getValue().stream()
                                        .map(duOrStageName ->
                                                YTree.mapBuilder()
                                                        .key(date).value(daysInUnixTime)
                                                        .key(statisticsName).value(statNameToGroupsByValues.getKey())
                                                        .key(statisticsValue).value(statValueToDuOrStageName.getKey())
                                                        .key(StatisticsService.duOrStageName).value(duOrStageName)
                                                        .buildMap()
                                        )
                                )
                        ).collect(Collectors.toList())
        );
    }

    private void writeToYtTable(String tablePath, boolean isAppend, List<YTreeMapNode> rows) {
        ytTables.write(
                YPath.simple(tablePath).append(isAppend),
                YTableEntryTypes.YSON,
                rows
        );
    }
}
