package ru.yandex.infra.sidecars_updater.sandbox;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Spliterators;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Charsets;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.ListenableFuture;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SandboxClientImpl implements SandboxClient {
    private static final Logger LOG = LoggerFactory.getLogger(SandboxClientImpl.class);

    private final AsyncHttpClient httpClient;
    private final String urlPrefix;
    private final String token;
    private final ObjectMapper mapper = new ObjectMapper();

    public SandboxClientImpl(AsyncHttpClient httpClient, String baseUrl, String token) {
        this.httpClient = httpClient;
        this.urlPrefix = String.format("%s/api/v1.0/resource", baseUrl);
        this.token = token;
    }

    private String getAttrsString(Map<String, String> attributes) {
        ObjectNode attrsNode = mapper.createObjectNode();
        attributes.forEach(attrsNode::put);
        try {
            return URLEncoder.encode(mapper.writeValueAsString(attrsNode), Charsets.UTF_8.name());
        } catch (UnsupportedEncodingException | JsonProcessingException e) {
            LOG.error(e.getMessage());
            throw new RuntimeException(e);
        }
    }

    @Override
    public CompletableFuture<List<SandboxResourceInfo>> getResources(
            String type, Map<String, String> attributes, boolean hidden, long limit
    ) {
        String attrsString = getAttrsString(attributes);
        String url = String.format("%s?type=%s&attrs=%s&state=READY&hidden=%b&limit=%d&order=-id",
                urlPrefix, type, attrsString, hidden, limit);
        return doRequestThenExtract(
                url,
                root -> streamFromIterator(root.get("items").elements())
                        .map(item -> new SandboxResourceInfo(
                                item.get("id").asLong(),
                                item.get("type").asText(),
                                item.get("skynet_id").asText(),
                                item.get("http").get("proxy").asText(),
                                streamFromIterator(item.get("http").get("links").elements())
                                        .map(JsonNode::asText)
                                        .collect(Collectors.toList()))
                        ).collect(Collectors.toList())
        );
    }

    private <T> Stream<T> streamFromIterator(Iterator<T> iterator) {
        return StreamSupport.stream(Spliterators.spliteratorUnknownSize(iterator, 0), false);
    }

    private <T> CompletableFuture<T> doRequestThenExtract(String url, Function<JsonNode, T> extractor) {
        try {
            return doGETRequest(url).toCompletableFuture()
                    .thenApply(response -> {
                        try {
                            byte[] responseBodyBytes = response.getResponseBodyAsBytes();
                            JsonNode root = mapper.readTree(responseBodyBytes);
                            return extractor.apply(root);
                        } catch (IOException e) {
                            throw new RuntimeException("IO Error while getting response data", e);
                        } catch (RuntimeException e) {
                            throw new RuntimeException("Can't extract data", e);
                        }
                    });
        } catch (Exception e) {
            LOG.error("Sandbox request setup failed", e);
            return CompletableFuture.failedFuture(e);
        }
    }

    private ListenableFuture<Response> doGETRequest(String url) {
        return httpClient.executeRequest(httpClient
                .prepareGet(url)
                .addHeader("Authorization", String.format("OAuth %s", token))
                .build()
        );
    }
}
