package ru.yandex.infra.sidecars_updater.sandbox;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SandboxInfoGetterImpl implements SandboxInfoGetter {
    private static final Logger LOG = LoggerFactory.getLogger(SandboxClientImpl.class);

    private final SandboxClient sandboxClient;
    private final Duration timeout;
    private final long resourcesSizeLimit;
    private final ConcurrentMap<String, SandboxResourceInfo> infoByUrlCache;

    @VisibleForTesting
    SandboxInfoGetterImpl(SandboxClient sandboxClient, Duration timeout, long resourcesSizeLimit,
                          ConcurrentMap<String, SandboxResourceInfo> infoByUrlCache) {
        this.sandboxClient = sandboxClient;
        this.timeout = timeout;
        this.resourcesSizeLimit = resourcesSizeLimit;
        this.infoByUrlCache = infoByUrlCache;
    }

    public SandboxInfoGetterImpl(SandboxClient sandboxClient, Duration timeout, long resourcesSizeLimit) {
        this(sandboxClient, timeout, resourcesSizeLimit, new ConcurrentHashMap<>());
    }

    public void refreshTypeCache(String type, Map<String, String> attributes, boolean hidden) {
        sandboxClient.getResources(type, attributes, hidden, resourcesSizeLimit)
                .thenAccept(resources -> {
                            resources.forEach(resource -> {
                                infoByUrlCache.putIfAbsent(resource.getSbrUrl(), resource);
                                infoByUrlCache.putIfAbsent(resource.getSkynetId(), resource);
                                infoByUrlCache.putIfAbsent(resource.getHttpProxyUrl(), resource);
                                resource.getHttpUrls().forEach(url -> infoByUrlCache.putIfAbsent(url, resource));
                            });
                            LOG.info("Updated caches for resource type {}", type);
                        }
                )
                .orTimeout(timeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    @Override
    public Optional<SandboxResourceInfo> getSandboxResourceInfoByUrl(String url) {
        return Optional.ofNullable(infoByUrlCache.get(url));
    }

    @VisibleForTesting
    ConcurrentMap<String, SandboxResourceInfo> getInfoByUrlCache() {
        return infoByUrlCache;
    }
}
