package ru.yandex.infra.sidecars_updater.servlets;

import java.io.IOException;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsService;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsServiceProxy;
import ru.yandex.infra.sidecars_updater.sidecar_service.SidecarsServiceProxyError;
import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;
import ru.yandex.infra.sidecars_updater.webauth.WebAuthClient;
import ru.yandex.misc.lang.StringUtils;

import static java.lang.String.format;

public class ApiServlet extends HttpServlet {
    public static final int STATISTIC_TIMEOUT_IN_SECONDS = 30;
    private static final Logger LOG = LoggerFactory.getLogger(ApiServlet.class);
    private static final String REQUEST_PARAMETER_PERCENT = "percent";
    private static final String REQUEST_PARAMETER_SIDECAR = "sidecar";
    private static final String REQUEST_PARAMETER_PATCHERS_REVISION = "patchers-revision";
    private static final String REQUEST_PARAMETER_UPDATE_ID = "update-id";

    private final SidecarsService sidecarsService;
    private final SidecarsServiceProxy sidecarsServiceProxy;
    private final Operation operation;
    private final WebAuthClient webAuthClient;

    public ApiServlet(SidecarsService sidecarsService,
                      SidecarsServiceProxy sidecarsServiceProxy,
                      Operation operation,
                      WebAuthClient webAuthClient) {
        this.sidecarsService = sidecarsService;
        this.sidecarsServiceProxy = sidecarsServiceProxy;
        this.operation = operation;
        this.webAuthClient = webAuthClient;
    }

    private static OptionalInt parsePercent(String percentParam) {
        if (!StringUtils.isNumericArabic(percentParam)) {
            return OptionalInt.empty();
        }

        int percent = Integer.parseInt(percentParam);
        if (percent < 0 || percent > 100) {
            return OptionalInt.empty();
        }
        return OptionalInt.of(percent);
    }

    private static String getParam(JsonNode node, String path) {
        if (node.has(path)) {
            var n = node.get(path);
            if (n.isTextual()) {
                return n.textValue();
            }
            if (n.isNumber()) {
                return n.numberValue().toString();
            }
        }
        return null;
    }

    @Override
    public void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        enableCORS(req, resp);
        final long start = System.currentTimeMillis();

        String response;
        switch (operation) {
            case LIST_SIDECAR_RELEASES:
                response = doListSidecars();
                break;
            case PATCHER_STATISTIC:
                response = doSidecarStatistic(Optional.empty());
                break;
            case SIDECAR_STATISTIC:
                if (req.getParameter(REQUEST_PARAMETER_SIDECAR) == null) {
                    response = "sidecar must be specified\n";
                } else {
                    response = doSidecarStatistic(Optional.of(req.getParameter(REQUEST_PARAMETER_SIDECAR)));
                }
                break;
            case SIDECAR_UPDATE:
                if (req.getParameter(REQUEST_PARAMETER_UPDATE_ID) == null) {
                    response = "update-id must be specified\n";
                } else {
                    try {
                        response = sidecarsServiceProxy.getStatus(req.getParameter(REQUEST_PARAMETER_UPDATE_ID));
                    } catch (SidecarsServiceProxyError e) {
                        response = e.getMessage();
                    }
                }
                break;
            default:
                response = format("Unexpected GET operation %s\n", operation);
        }
        resp.getWriter().print(response);
        LOG.info("Processing GET {} request took: {} ms", operation, System.currentTimeMillis() - start);
    }

    @Override
    public void doPost(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        enableCORS(req, resp);
        ObjectMapper mapper = new ObjectMapper();
        JsonNode reqBody = mapper.readTree(IOUtils.toString(req.getReader()));

        if (!isAuthorized(req, resp, getParam(reqBody, REQUEST_PARAMETER_SIDECAR))) {
            return;
        }
        final long start = System.currentTimeMillis();

        String initiator;
        try {
            initiator = webAuthClient.getUserLogin(req.getHeader("Cookie"))
                    .get(3, TimeUnit.SECONDS);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            e.printStackTrace();
            resp.sendError(403);
            return;
        }

        String response;
        switch (operation) {
            case APPLY:
            case APPLY_SIDECAR:
                response = doApply(getParam(reqBody, REQUEST_PARAMETER_PERCENT),
                        getParam(reqBody, REQUEST_PARAMETER_SIDECAR),
                        getParam(reqBody, REQUEST_PARAMETER_PATCHERS_REVISION),
                        initiator);
                break;
            default:
                response = format("Unexpected POST operation %s\n", operation);
        }

        resp.getWriter().print(response);
        LOG.info("Processing POST {} request took: {} ms", operation, System.currentTimeMillis() - start);
    }

    @Override
    protected void doOptions(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        enableCORS(req, resp);
        resp.addHeader("Access-Control-Allow-Headers", "Content-Type");
        super.doOptions(req, resp);
    }

    private String doListSidecars() {
        String response;
        try {
            sidecarsService.refreshData();
            StringBuilder builder = new StringBuilder();
            sidecarsService.getSidecars().forEach(builder::append);
            response = builder + "\n";
        } catch (RuntimeException e) {
            LOG.error("Failed to get list of sidecars:", e);
            response = "Failed\n";
        }
        return response;
    }

    private String doSidecarStatistic(Optional<String> sidecarTypeName) {
        Optional<Sidecar.Type> sidecarType = Optional.empty();
        if (sidecarTypeName.isPresent()) {
            try {
                sidecarType = Optional.of(Sidecar.Type.valueOf(sidecarTypeName.get()));
            } catch (IllegalArgumentException e) {
                return "invalid sidecar param\n";
            }
        }

        String resp;
        try {
            var infos = sidecarsService
                    .collectStatistics(sidecarType)
                    .get(STATISTIC_TIMEOUT_IN_SECONDS, TimeUnit.SECONDS);
            resp = new ObjectMapper().writeValueAsString(infos);
        } catch (ExecutionException | TimeoutException | InterruptedException | JsonProcessingException e) {
            e.printStackTrace();
            resp = "Error: " + e.getMessage();
        }

        return resp;
    }

    private String doApply(String percentParam,
                           String sidecarTypeName,
                           String patchersRevisionParam,
                           String initiator) {
        OptionalInt percentOpt = parsePercent(percentParam);
        if (percentOpt.isEmpty()) {
            return "invalid percent param\n";
        }

        if (sidecarTypeName == null && patchersRevisionParam == null) {
            return "sidecar or patchers revision must be specified\n";
        }

        int percent = percentOpt.getAsInt();
        Optional<Sidecar.Type> sidecarTypeOpt;
        if (sidecarTypeName != null) {
            try {
                sidecarTypeOpt = Optional.of(Sidecar.Type.valueOf(sidecarTypeName));
            } catch (IllegalArgumentException e) {
                return "invalid sidecar param\n";
            }
        } else {
            sidecarTypeOpt = Optional.empty();
        }

        OptionalInt patchersRevisionPercentOpt = OptionalInt.empty();
        if (patchersRevisionParam != null) {
            if (!StringUtils.isNumericArabic(patchersRevisionParam)) {
                return "invalid patchers revision\n";
            }
            patchersRevisionPercentOpt = OptionalInt.of(Integer.parseInt(patchersRevisionParam));
        }

        Optional<Sidecar> sidecar = sidecarsService.getSidecars().stream()
                .filter(s -> sidecarTypeOpt.stream().anyMatch(type -> s.getResourceType().equals(type)))
                .findFirst();

        if (sidecarTypeOpt.isPresent() && sidecar.isEmpty()) {
            return "Sidecar with type" + sidecarTypeName + "not found. Available sidecars: " +
                    sidecarsService.getSidecars().stream()
                            .map(Sidecar::getResourceType)
                            .map(Enum::toString)
                            .collect(Collectors.joining(", "))
                    + '\n';
        }
        try {
            Pair<String, String> idAndRef = sidecarsServiceProxy.applyOnPercent(sidecar, patchersRevisionPercentOpt,
                    percent, initiator);
            return "Task started with id: " + idAndRef.getLeft() + "\n\n" +
                    "Ticket: " + idAndRef.getRight() + "\n";
        } catch (SidecarsServiceProxyError e) {
            return e.getMessage();
        }
    }

    private boolean isAuthorized(HttpServletRequest req, HttpServletResponse resp, String sidecarTypeName) throws IOException {
        try {
            String userLogin = webAuthClient.getUserLogin(req.getHeader("Cookie"))
                    .get(3, TimeUnit.SECONDS);
            if (userLogin.isEmpty()) {
                resp.sendError(403);
                return false;
            }
            LOG.info("new request from {}", userLogin);
            if (webAuthClient.isUserWhitelisted(userLogin)) {
                return true;
            }
            var sidecar = Sidecar.Type.valueOf(sidecarTypeName);
            if (!webAuthClient.isUserHasAccessToSidecar(userLogin, sidecar)) {
                resp.sendError(403);
                return false;
            }
        } catch (InterruptedException | ExecutionException | TimeoutException | IllegalArgumentException e) {
            e.printStackTrace();
            resp.sendError(403);
            return false;
        }
        return true;
    }

    private void enableCORS(HttpServletRequest req, HttpServletResponse resp) {
        String origin = req.getHeader("origin");
        if (null == origin) {
            origin = "*";
        }
        resp.addHeader("Access-Control-Allow-Origin", origin);
        resp.addHeader("Access-Control-Allow-Credentials", "true");
    }

    public enum Operation {
        PATCHER_STATISTIC,
        SIDECAR_STATISTIC,
        LIST_SIDECAR_RELEASES,
        @Deprecated
        APPLY_SIDECAR,
        APPLY,
        SIDECAR_UPDATE
    }
}
