package ru.yandex.infra.sidecars_updater.sidecar_service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.sidecars_updater.DeployUnitSidecarInfo;
import ru.yandex.infra.sidecars_updater.LabelUpdater;
import ru.yandex.infra.sidecars_updater.StageCache;
import ru.yandex.infra.sidecars_updater.StageUpdateNotifier;
import ru.yandex.infra.sidecars_updater.StageUpdateNotifierError;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxClient;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxInfoGetter;
import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

public class SidecarsService {
    // Labels for UI
    public static final String DU_SIDECAR_TARGET_LABEL = "du_sidecar_target_revision";
    public static final String DU_PATCHERS_TARGET_LABEL = "du_patchers_target_revision";
    // Autoupdate labels
    public static final String DU_SIDECAR_AUTOUPDATE_LABEL = "du_sidecar_autoupdate_revision";
    public static final String DU_PATCHERS_AUTOUPDATE_LABEL = "du_patchers_autoupdate_revision";

    @VisibleForTesting
    static final List<Sidecar.Type> CACHED_SIDECAR_TYPES = List.of(
            Sidecar.Type.PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP,
            Sidecar.Type.PORTO_LAYER_SEARCH_UBUNTU_BIONIC_APP,
            Sidecar.Type.PORTO_LAYER_SEARCH_UBUNTU_FOCAL_APP
    );

    private static final Logger LOG = LoggerFactory.getLogger(SidecarsService.class);

    private static final String UNKNOWN_REVISION = "default";
    private final LabelUpdater<TStageSpec, TStageStatus> labelUpdater;
    private final SandboxClient sandboxClient;
    private final SandboxInfoGetter sandboxInfoGetter;
    private final StageUpdateNotifier stageUpdateNotifier;
    private final Set<String> blackList;
    private final Set<String> whiteList;
    private final List<Sidecar> sidecars;
    private final StageCache stageCache;

    public SidecarsService(LabelUpdater<TStageSpec, TStageStatus> labelUpdater,
                           List<Sidecar> sidecars, SandboxClient sandboxClient,
                           SandboxInfoGetter sandboxInfoGetter,
                           StageUpdateNotifier stageUpdateNotifier,
                           Set<String> blackList,
                           Set<String> whiteList,
                           StageCache stageCache) {
        this.labelUpdater = labelUpdater;
        this.sidecars = sidecars;
        this.sandboxClient = sandboxClient;
        this.sandboxInfoGetter = sandboxInfoGetter;
        this.stageUpdateNotifier = stageUpdateNotifier;
        this.blackList = blackList;
        this.whiteList = whiteList;
        this.stageCache = stageCache;
    }

    public CompletableFuture<List<DeployUnitSidecarInfo>> collectDeployUnitData() {
        return stageCache.getStagesFutureCache()
                .thenApply(stages -> {
                    stages.values().forEach(stage -> labelUpdater.refreshLabelsInfo(stage.getMeta().getId(),
                            stage.getLabels()));
                    return stages;
                })
                .thenApply(stages -> stages.values().stream()
                        .flatMap(stage -> stage.getSpec().getDeployUnitsMap().keySet().stream().map(name -> Pair.of(stage, name)))
                        .flatMap(pair -> sidecars.stream().map(sidecar -> new DeployUnitSidecarInfo(pair.getLeft(),
                                pair.getRight(), sidecar)))
                        .filter(DeployUnitSidecarInfo::isUsed)
                        .collect(Collectors.toList())
                );
    }

    public CompletableFuture<Map<String, SortedSet<String>>> collectStatistics(Optional<Sidecar.Type> sidecarType) {
        return collectDeployUnitData().thenApply(infos -> infos.stream()
                .filter(s -> sidecarType.isEmpty() || s.getSidecarType().equals(sidecarType.get()))
                .collect(Collectors.groupingBy((info) -> {
                            if (sidecarType.isEmpty()) {
                                return Optional.of(info.getPatchersRevision()).filter(x -> x > 0).map(Object::toString).orElse(UNKNOWN_REVISION);
                            }
                            return info.getCurrentRev().filter(x -> x > 0).map(Object::toString).orElse(UNKNOWN_REVISION);
                        },
                        Collectors.mapping(DeployUnitSidecarInfo::getFullDeployUnitName,
                                Collectors.toCollection(TreeSet::new))))
        );
    }

    public CompletableFuture<Pair<List<String>, List<String>>> applyOnPercent(Optional<Sidecar> sidecar,
                                                                              OptionalInt patchersRevision,
                                                                              int percent, String initiator) {
        return collectDeployUnitData()
                .thenCompose(infos -> {
                    List<DeployUnitSidecarInfo> toUpdate = new ArrayList<>();
                    AtomicInteger total = new AtomicInteger();
                    infos.stream()
                            .filter(info -> checkBlackWhiteList(info.getStageName()))
                            .forEach(info -> {
                                boolean isSidecarTypeMatches =
                                        sidecar.stream().anyMatch(s -> info.getSidecarType().equals(s.getResourceType()));
                                boolean isSidecarUpdateRequired =
                                        sidecar.stream().anyMatch(s -> sidecarUpdateRequired(info, s)) && isSidecarTypeMatches;
                                boolean isPatchersUpdateRequired =
                                        patchersRevision.stream().anyMatch(r -> patcherRevisionRequired(info, r));

                                if (isSidecarTypeMatches || isPatchersUpdateRequired) {
                                    total.getAndIncrement();
                                }
                                if (isSidecarUpdateRequired || isPatchersUpdateRequired) {
                                    toUpdate.add(info);
                                }
                            });
                    int shouldBeUpdated = total.get() * percent / 100;
                    int alreadyUpdated = total.get() - toUpdate.size();
                    int toUpdateCnt = Math.min(Math.max(shouldBeUpdated - alreadyUpdated, 0), toUpdate.size());

                    String description = Stream.concat(
                            sidecar.map(Sidecar::getResourceType).map(s -> "resource type: " + s).stream(),
                            patchersRevision.stream().mapToObj(r -> "patchers revision: " + r)
                    ).collect(Collectors.joining(", "));

                    LOG.info("apply {} for {} of {} deploy units. {} already updated.", description,
                            toUpdateCnt, total.get(), alreadyUpdated);

                    Collections.shuffle(toUpdate);
                    return updateSidecarsAndPatchersRevision(
                            toUpdate.subList(0, toUpdateCnt), sidecar, patchersRevision, initiator
                    );
                });
    }

    private CompletableFuture<Pair<List<String>, List<String>>> updateSidecarsAndPatchersRevision(List<DeployUnitSidecarInfo> deployUnitsToUpdate,
                                                                                                  Optional<Sidecar> sidecar,
                                                                                                  OptionalInt patchersRevision,
                                                                                                  String initiator) {
        List<String> successes = new ArrayList<>();
        List<String> errors = new ArrayList<>();
        CompletableFuture[] clf = deployUnitsToUpdate.stream().map(info -> {
            String stageName = info.getStageName();
            final Map<String, YTreeNode> patches = new HashMap<>();

            if (info.isEnableAutoUpdate()) {
                sidecar.ifPresent(s -> patches.put(DU_SIDECAR_AUTOUPDATE_LABEL, buildLabels(info, s)));
                patchersRevision.ifPresent(rev -> patches.put(DU_PATCHERS_AUTOUPDATE_LABEL, buildPatchersLabels(info,
                        rev)));
            } else {
                sidecar.ifPresent(s -> patches.put(DU_SIDECAR_TARGET_LABEL, buildLabels(info, s)));
                patchersRevision.ifPresent(rev -> patches.put(DU_PATCHERS_TARGET_LABEL, buildPatchersLabels(info,
                        rev)));
            }

            return labelUpdater.patchLabels(stageName, patches)
                    .whenComplete((res, e) -> {
                        if (e != null) {
                            LOG.info("error {}", e.getMessage());
                            errors.add(info.getFullDeployUnitName());
                        } else {
                            LOG.info("Updated {} {}", stageName, res);
                            successes.add(info.getFullDeployUnitName());
                            try {
                                if (sidecar.isPresent() && !info.isEnableAutoUpdate()) {
                                    stageUpdateNotifier.updateNotify(info.getStageMeta(), sidecars);
                                }
                            } catch (StageUpdateNotifierError err) {
                                LOG.error("Error while notifying owners of stage {}: ", info.getStageName(), err);
                            }
                        }
                    });
        }).toArray(CompletableFuture[]::new);

        String description = Stream.concat(
                sidecar.map(s -> String.format("%s(rev: %d)", s.getResourceType(), s.getRevision())).stream(),
                patchersRevision.stream().mapToObj(r -> "patchers(rev: " + r + ')')
        ).collect(Collectors.joining(", "));

        return CompletableFuture.allOf(clf)
                .handle((res, err) -> {
                    LOG.info(String.format("Applied %s to %d deploy unit(s). Updated: %d, yp errors: %d.\n",
                            description, deployUnitsToUpdate.size(), successes.size(), errors.size()));
                    return Pair.of(successes, errors);
                });
    }

    private boolean checkBlackWhiteList(String name) {
        if (whiteList.isEmpty()) {
            return !blackList.contains(name);
        }
        return whiteList.contains(name);
    }

    private boolean patcherRevisionRequired(DeployUnitSidecarInfo info, int patchersRevision) {
        if (info.getTargetPatchersRevision().isPresent()) {
            return info.getTargetPatchersRevision().get() != patchersRevision;
        }
        return info.getPatchersRevision() != patchersRevision;
    }

    private boolean sidecarUpdateRequired(DeployUnitSidecarInfo info, Sidecar sidecar) {
        Optional<Long> target = info.getTargetRev();

        if (target.isPresent()) {
            return target.get() != sidecar.getRevision();
        }

        Optional<Long> currentRevision = info.getCurrentRev();
        if (currentRevision.isEmpty()) {
            return true;
        }

        return currentRevision.get() != sidecar.getRevision();
    }

    private YTreeNode buildLabels(DeployUnitSidecarInfo info, Sidecar sidecar) {
        YTreeBuilder builder = new YTreeBuilder().beginMap();
        builder.key(info.getDeployUnitName()).value(new YTreeBuilder()
                .beginMap()
                .key(sidecar.getLabelName()).value(sidecar.getRevision())
                .endMap()
                .build()
        );
        builder.endMap();
        return builder.build();
    }

    private YTreeNode buildPatchersLabels(DeployUnitSidecarInfo info, int revision) {
        return new YTreeBuilder()
                .beginMap()
                .key(info.getDeployUnitName()).value(revision)
                .endMap()
                .build();
    }

    /**
     * Refreshes needing information in <var>sidecars</var> and stages from YP.
     */
    public void refreshData() {
        stageCache.refreshStageCache();
        refreshSandboxCaches();
        collectLastReleases();
    }

    /**
     * Refreshes caches in <var>sandboxResourceInfoGetter</var>.
     * It's important for sidecars of {@link SidecarsService#CACHED_SIDECAR_TYPES} types.
     */
    private void refreshSandboxCaches() {
        for (Sidecar sidecar : sidecars) {
            if (CACHED_SIDECAR_TYPES.contains(sidecar.getResourceType())) {
                sandboxInfoGetter.refreshTypeCache(
                        sidecar.getResourceType().toString(),
                        sidecar.getAttributes(),
                        sidecar.considerHidden()
                );
            }
        }
    }

    /**
     * Collects last releases of <a href="https://sandbox.yandex-team.ru/resources">Sandbox</a> resources
     * of {@link Sidecar.Type} types.
     */
    private void collectLastReleases() {
        sidecars.forEach(sidecar -> sandboxClient.getResources(
                                sidecar.getResourceType().toString(), sidecar.getAttributes(),
                                sidecar.considerHidden(), 1
                        )
                        .thenApply(list -> list.get(0).getRevision())
                        .thenAccept(result -> {
                            LOG.info("Resolved resource {} - {}", sidecar.getResourceType(), result);
                            sidecar.setRevision(result);
                        })
                        .orTimeout(1, TimeUnit.MINUTES)
        );
    }

    public List<Sidecar> getSidecars() {
        return ImmutableList.copyOf(sidecars);
    }

    @VisibleForTesting
    LabelUpdater<TStageSpec, TStageStatus> getLabelUpdater() {
        return labelUpdater;
    }

    @VisibleForTesting
    SandboxClient getSandboxClient() {
        return sandboxClient;
    }

    @VisibleForTesting
    SandboxInfoGetter getSandboxInfoGetter() {
        return sandboxInfoGetter;
    }

    @VisibleForTesting
    StageCache getStageCache() {
        return stageCache;
    }

}
