package ru.yandex.infra.sidecars_updater.sidecar_service;

import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;

import ru.yandex.infra.sidecars_updater.StageUpdateNotifier;
import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.startrek.client.model.Issue;

public class UpdateTask {
    public static final String STATUS = "status";
    public static final String ID = "id";
    public static final String SIDECAR = "sidecar";
    public static final String PATCHER = "patcher";
    public static final String PERCENT = "percent";
    public static final String INITIATOR = "initiator";
    public static final String ATTEMPTS = "attempts";
    private final String id;
    private final Optional<Sidecar> sidecar;
    private final OptionalInt patcher;
    private final int percent;
    private final String initiator;
    private int attempts;
    private UpdateTask.Status status;
    private final Optional<Issue> issue;


    UpdateTask(String id, Optional<Sidecar> sidecar, OptionalInt patcher, int percent, String initiator,
               Optional<Issue> issue) {
        this.id = id;
        this.sidecar = sidecar;
        this.patcher = patcher;
        this.percent = percent;
        this.initiator = initiator;
        this.issue = issue;
        attempts = 0;
        status = UpdateTask.Status.IN_PROGRESS;
    }

    public UpdateTask(UpdateTaskDTO dto, SidecarsService sidecarsService, StageUpdateNotifier stageUpdateNotifier) {
        id = dto.getId();
        sidecar = prepareSidecar(dto.getSidecar(), sidecarsService);
        patcher = dto.getPatcher() == 0 ? OptionalInt.of(dto.getPatcher()) : OptionalInt.empty();
        percent = dto.getPercent();
        initiator = dto.getInitiator();
        attempts = dto.getAttempts();
        status = UpdateTask.Status.valueOf(dto.getStatus());
        if (status == UpdateTask.Status.IN_PROGRESS) {
            issue = stageUpdateNotifier.getIssue(id);
        } else {
            issue = Optional.empty();
        }
    }

    public UpdateTaskDTO toUpdateTaskDTO() {
        return new UpdateTaskDTO(id,
                sidecar.map(value -> value.getResourceType().name()).orElse(null),
                patcher.isPresent() ? patcher.getAsInt() : null,
                percent, initiator, attempts, status.name());
    }

    public YTreeMapNode toYTreeMap() {
        var requestBuilder =
                YTree.mapBuilder()
                        .key(ID).value(id)
                        .key(PERCENT).value(percent)
                        .key(INITIATOR).value(initiator)
                        .key(ATTEMPTS).value(attempts)
                        .key(STATUS).value(status.name());
        if (sidecar.isPresent()) {
            requestBuilder = requestBuilder.key(SIDECAR).value(sidecar.get().getResourceType().name());
        }
        if (patcher.isPresent()) {
            requestBuilder = requestBuilder.key(PATCHER).value(patcher.getAsInt());
        }
        return requestBuilder.buildMap();
    }

    private Optional<Sidecar> prepareSidecar(String sidecarName, SidecarsService sidecarsService) {
        Optional<Sidecar.Type> sidecarTypeOpt;
        if (sidecarName != "") {
            try {
                sidecarTypeOpt = Optional.of(Sidecar.Type.valueOf(sidecarName));
            } catch (IllegalArgumentException e) {
                throw new RuntimeException("invalid sidecar param\n", e);
            }
        } else {
            sidecarTypeOpt = Optional.empty();
        }
        return sidecarsService.getSidecars().stream()
                .filter(s -> sidecarTypeOpt.stream()
                        .anyMatch(type -> s.getResourceType().equals(type)))
                .findFirst();
    }

    public Optional<Sidecar> getSidecar() {
        return sidecar;
    }

    public OptionalInt getPatcher() {
        return patcher;
    }

    public String getInitiator() {
        return initiator;
    }

    public int getAttempts() {
        return attempts;
    }

    public Status getStatus() {
        return status;
    }

    public void setStatus(Status status) {
        this.status = status;
    }

    public void incrementAttempts() {
        attempts++;
    }

    public String getId() {
        return id;
    }

    public int getPercent() {
        return percent;
    }

    public Optional<Issue> getIssue() {
        return issue;
    }

    public boolean isSuccess() {
        return status == UpdateTask.Status.SUCCESS;
    }

    @Override
    public String toString() {
        return "UpdateState{" +
                "id='" + id + '\'' +
                ", sidecar=" + sidecar +
                ", patcher=" + patcher +
                ", percent=" + percent +
                ", initiator='" + initiator + '\'' +
                ", attempts=" + attempts +
                ", status=" + status +
                ", issue=" + issue +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof UpdateTask) {
            UpdateTask that = (UpdateTask) o;
            return id.equals(that.id);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    enum Status {
        SUCCESS("SUCCESS"), IN_PROGRESS("IN_PROGRESS"), FAILED("FAILED");
        private final String status;

        Status(String status) {
            this.status = status;
        }

        @Override
        public String toString() {
            return status;
        }
    }
}
