package ru.yandex.infra.sidecars_updater.sidecar_service;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.sidecars_updater.StageUpdateNotifier;

public class UpdateTaskRunner {
    private static final Logger LOG = LoggerFactory.getLogger(UpdateTaskRunner.class);
    public static final int FULL_PERCENT = 100;
    private static final int UPDATE_TIMEOUT_IN_SECONDS = 30;
    private final Map<String, UpdateTask> localTasks = new ConcurrentHashMap<>();
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final SidecarsService sidecarsService;
    private final StageUpdateNotifier stageUpdateNotifier;
    private final YtUpdateTaskRepository ytUpdateTaskRepository;
    private final int attemptsLimit;
    public final Duration workerCycleSleep;

    public UpdateTaskRunner(SidecarsService sidecarsService, StageUpdateNotifier stageUpdateNotifier,
                            YtUpdateTaskRepository ytUpdateTaskRepository, int attemptsLimit,
                            Duration workerCycleSleep) {
        this.sidecarsService = sidecarsService;
        this.stageUpdateNotifier = stageUpdateNotifier;
        this.ytUpdateTaskRepository = ytUpdateTaskRepository;
        this.attemptsLimit = attemptsLimit;
        this.workerCycleSleep = workerCycleSleep;

    }

    public void addTask(UpdateTask updateTask) {
        localTasks.put(updateTask.getId(), updateTask);
    }

    public Optional<UpdateTask> getTask(String id) {
        return Optional.ofNullable(localTasks.get(id));
    }


    public void run() {
        Runnable task = () -> {
            AtomicInteger successCounter = new AtomicInteger(localTasks.size());

            localTasks.values().forEach(state -> {
                if (state.getStatus() == UpdateTask.Status.IN_PROGRESS) {
                    try {
                        var ans = sidecarsService
                                .applyOnPercent(state.getSidecar(), state.getPatcher(), state.getPercent(),
                                        state.getInitiator())
                                .get(UPDATE_TIMEOUT_IN_SECONDS, TimeUnit.SECONDS);
                        var successes = ans.getLeft();
                        var errors = ans.getRight();
                        if (errors.isEmpty()) {
                            state.setStatus(UpdateTask.Status.SUCCESS);
                        } else {
                            state.incrementAttempts();
                            if (state.getAttempts() >= attemptsLimit) {
                                if (state.getPercent() == FULL_PERCENT) {
                                    state.getIssue().ifPresent(issue -> stageUpdateNotifier.commentIssue(issue,
                                            "FAILED FOR: " + errors));
                                }
                                state.setStatus(UpdateTask.Status.FAILED);
                            }
                        }
                        if (successes.isEmpty()) {
                            successCounter.getAndIncrement();
                        } else {
                            state.getIssue().ifPresent(issue -> stageUpdateNotifier.commentIssue(issue,
                                    "SUCCESS FOR: " + successes));
                        }
                    } catch (InterruptedException | ExecutionException | TimeoutException e) {
                        LOG.error("Error during sidecar applying {} with id={}", e.getMessage(), state.getId());
                        state.incrementAttempts();
                        if (state.getAttempts() >= attemptsLimit) {
                            state.getIssue().ifPresent(issue -> stageUpdateNotifier.commentIssue(issue,
                                    "Exception with message: " + e.getMessage()));
                            state.setStatus(UpdateTask.Status.FAILED);
                        }
                    } catch (Exception e) {
                        state.getIssue().ifPresent(issue -> stageUpdateNotifier.commentIssue(issue,
                                "Exception with message: " + e.getMessage()));
                        state.setStatus(UpdateTask.Status.FAILED);
                    }
                }
            });
            try {
                ytUpdateTaskRepository.insertRows(new ArrayList<>(localTasks.values()));
                List<String> idsToRemove =
                        localTasks.values().stream()
                                .filter(it -> it.getStatus() != UpdateTask.Status.IN_PROGRESS)
                                .map(it -> {
                                    it.getIssue().ifPresent(issue -> stageUpdateNotifier.updateIssue(issue,
                                            "Ended with status: " + it.getStatus()));
                                    return it.getId();
                                })
                                .collect(Collectors.toList());
                idsToRemove.forEach(localTasks::remove);
            } catch (Exception e) {
                LOG.error("Exception on writing to YP", e);
            }
        };
        executor.scheduleWithFixedDelay(task, 1, workerCycleSleep.toMillis(), TimeUnit.MILLISECONDS);
    }

    @VisibleForTesting
    public Map<String, UpdateTask> getLocalTasks() {
        return localTasks;
    }

    @VisibleForTesting
    public SidecarsService getSidecarsService() {
        return sidecarsService;
    }

    @VisibleForTesting
    public StageUpdateNotifier getStageUpdateNotifier() {
        return stageUpdateNotifier;
    }
    @VisibleForTesting
    public YtUpdateTaskRepository getYtUpdateTaskController() {
        return ytUpdateTaskRepository;
    }

}
