package ru.yandex.infra.sidecars_updater.sidecar_service;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.infra.sidecars_updater.StageUpdateNotifier;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializerFactory;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.proxy.SelectRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.request.CreateNode;
import ru.yandex.yt.ytclient.proxy.request.StartTransaction;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

public class YtUpdateTaskRepository {
    private static final Logger LOG = LoggerFactory.getLogger(YtUpdateTaskRepository.class);
    private static final String UPDATE_STATE = "update_state";
    private static final TableSchema schema = new TableSchema.Builder()
            .addKey("id", ColumnValueType.STRING)
            .addValue("sidecar", ColumnValueType.STRING)
            .addValue("patcher", ColumnValueType.INT64)
            .addValue("percent", ColumnValueType.INT64)
            .addValue("initiator", ColumnValueType.STRING)
            .addValue("attempts", ColumnValueType.INT64)
            .addValue("status", ColumnValueType.STRING)
            .build();
    private final YtClient client;
    private final String tablePath;

    public YtUpdateTaskRepository(YtClient yt, String tableFolderPath) {
        this.client = yt;
        this.tablePath = tableFolderPath + "/" + UPDATE_STATE;
        try {
            if (!client.existsNode(tableFolderPath + "/" + UPDATE_STATE).get()) {
                try {
                    LOG.info("Creating table: {}", tablePath);
                    yt.createNode(new CreateNode(YPath.simple(tablePath), CypressNodeType.TABLE,
                            Cf.map("schema", schema.toYTree(), "dynamic", YTree.booleanNode(true))));

                    LOG.info("Mounting table: {}", tablePath);
                    yt.mountTable(tablePath, false).get(1, TimeUnit.SECONDS);
                    LOG.info("Successfully mounted table: {}", tablePath);

                } catch (RuntimeException e) {
                    try {
                        if (yt.existsNode(tablePath).get()) {
                            yt.removeNode(tablePath);
                        }
                    } catch (RuntimeException e1) {
                        LOG.error("Error in cleanup", e1);
                    }
                    throw e;
                }
            }
        } catch (ExecutionException | InterruptedException | TimeoutException e) {
            LOG.error("Failed to init yt table");
            throw new RuntimeException(e);
        }
    }

    private YPath getYPath() {
        return YPath.simple(tablePath);
    }

    public List<UpdateTask> selectAllRows(SidecarsService sidecarsService, StageUpdateNotifier stageUpdateNotifier) {
        SelectRowsRequest request = SelectRowsRequest.of(String.format("* FROM [%s]", tablePath));
        return client.selectRows(request)
                .thenApply(rowset -> rowset.getYTreeRows()
                        .stream()
                        .map(yTreeMapNode -> new UpdateTask(YTreeObjectSerializerFactory.forClass(UpdateTaskDTO.class).deserialize(yTreeMapNode),
                                sidecarsService, stageUpdateNotifier))
                        .collect(Collectors.toList())).join();
    }

    public void insertRows(Collection<UpdateTask> tasks) {
        if (tasks.isEmpty()) {
            return;
        }
        LOG.info("Writing {} tasks in table {}", tasks.size(), tablePath);
        ModifyRowsRequest request = new ModifyRowsRequest(tablePath, schema);
        tasks.forEach(task -> request.addInsert(task.toYTreeMap().asMap()));
        try (var transaction = client.startTransaction(StartTransaction.stickyMaster()).join()){
            transaction.modifyRows(request).join();
            transaction.commit().join();
            LOG.info("Transaction commited");
        }
    }

    public Optional<String> lookupRowStatus(String taskId) {
        UnversionedRowset rowset = client.lookupRows(new LookupRowsRequest(tablePath, schema.toLookup())
                .addFilter(taskId)
                .addLookupColumns(UpdateTask.STATUS)).join();
        if (rowset.getYTreeRows().isEmpty()) {
            return Optional.empty();
        } else {
            return rowset.getYTreeRows().get(0).getStringO(UpdateTask.STATUS);
        }
    }
}
