package ru.yandex.infra.sidecars_updater.sidecars;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import ru.yandex.infra.sidecars_updater.sandbox.SandboxInfoGetter;
import ru.yandex.infra.sidecars_updater.sandbox.SandboxResourceInfo;
import ru.yandex.yp.client.api.TDeployUnitSpec;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.pods.TLayer;

public class BoxLayerSidecar extends AbstractSidecar {

    private final SandboxInfoGetter sandboxInfoGetter;

    public BoxLayerSidecar(
            Map<Sidecar.Type, Map<String, String>> sidecarsAttributes, String label,
            Sidecar.Type type, SandboxInfoGetter sandboxInfoGetter
    ) {
        super(sidecarsAttributes, type, label);
        this.sandboxInfoGetter = sandboxInfoGetter;
    }

    @Override
    public Function<TDeployUnitSpec, Optional<Long>> getRevisionGetter() {
        return spec -> {
            List<TLayer> layers = getLayersList(spec);
            // Рассматриваем все слои в деплой юните, фильтруем слои только нашего типа и ищем старейший
            return layers.stream()
                    .map(TLayer::getUrl)
                    .map(sandboxInfoGetter::getSandboxResourceInfoByUrl)
                    .flatMap(Optional::stream)
                    .filter(this::isTypeMatched)
                    .map(SandboxResourceInfo::getRevision)
                    .min(Long::compareTo);
        };
    }

    @Override
    public boolean isUsedBy(TStageSpec spec, String deployUnitName) {
        if (spec.containsDeployUnits(deployUnitName)) {
            TDeployUnitSpec deployUnitSpec = spec.getDeployUnitsOrThrow(deployUnitName);
            List<TLayer> layers = getLayersList(deployUnitSpec);
            return layers.stream().anyMatch(
                    layer -> sandboxInfoGetter.getSandboxResourceInfoByUrl(layer.getUrl())
                            .map(this::isTypeMatched).orElse(false)
            );
        }
        return false;
    }

    private boolean isTypeMatched(SandboxResourceInfo info) {
        return getResourceType().toString().equals(info.getType());
    }

    private static List<TLayer> getLayersList(TDeployUnitSpec deployUnitSpec) {
        return getPodTemplateSpec(deployUnitSpec)
                .getSpec()
                .getPodAgentPayload()
                .getSpec()
                .getResources()
                .getLayersList();
    }

    private static TPodTemplateSpec getPodTemplateSpec(TDeployUnitSpec du) {
        return du.hasMultiClusterReplicaSet() ?
                du.getMultiClusterReplicaSet().getReplicaSet().getPodTemplateSpec() :
                du.getReplicaSet().getReplicaSetTemplate().getPodTemplateSpec();
    }

    @Override
    public boolean considerHidden() {
        return true;
    }
}
