package ru.yandex.infra.sidecars_updater.staff;

import java.io.IOException;
import java.util.concurrent.ExecutionException;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.lang.String.format;
import static org.apache.http.HttpStatus.SC_INTERNAL_SERVER_ERROR;
import static org.apache.http.HttpStatus.SC_OK;

public class StaffClientImpl implements StaffClient{
    private static final Logger LOG = LoggerFactory.getLogger(StaffClientImpl.class);

    private static final String API_URL_PATH = "https://staff-api.yandex-team.ru/v3";
    private static final String IS_PERSON_DISMISSED_QUERY = "persons?_fields=official.is_dismissed&login=";

    private final AsyncHttpClient httpClient;
    private final String token;
    private final ObjectMapper mapper = new ObjectMapper();

    public StaffClientImpl(AsyncHttpClient httpClient, String token) {
        this.httpClient = httpClient;
        this.token = token;
    }

    public boolean isPersonDismissed(String person) throws StaffClientError {
        String url = format("%s/%s%s", API_URL_PATH, IS_PERSON_DISMISSED_QUERY, person);

        Request request = httpClient.prepareGet(url)
                .addHeader("Authorization", format("OAuth %s", token))
                .build();

        try {
            return httpClient.executeRequest(request)
                    .toCompletableFuture()
                    .thenApply(this::processPersonDismissedResponse)
                    .get();
        } catch (InterruptedException | ExecutionException e) {
            String msg = String.format(
                    "Failed to get information about dismissed status of person %s: %s",
                    person,
                    e.getCause());
            LOG.error(msg);
            throw new StaffClientError(msg, e.getCause());
        }
    }

    private boolean processPersonDismissedResponse(Response response) {
        if (response.getStatusCode() == SC_OK) {
            try {
                IsDismissedResponse isDismissedResponse =
                        mapper.readValue(response.getResponseBody(), IsDismissedResponse.class);
                if (isDismissedResponse.getTotal() == 0) {
                    throw new RuntimeException("Did not find any staff person");
                } else if (isDismissedResponse.getTotal() != 1) {
                    throw new RuntimeException("Found more than one staff person");
                }
                return isDismissedResponse.getResults().get(0).getOfficial().isDismissed();
            } catch (IOException e) {
                throw new RuntimeException(
                        format("Error while parsing response from staff: %s", response.getStatusText()), e);
            }
        } else if (response.getStatusCode() >= SC_INTERNAL_SERVER_ERROR) {
            // server side error, need to retry later on
            throw new RuntimeException(String.format(
                    "Staff API error: %d [%s]",
                    response.getStatusCode(),
                    response.getStatusText()));
        }

        throw new RuntimeException(response.getStatusText());
    }
}
