package ru.yandex.infra.sidecars_updater.statistics;

import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import com.codahale.metrics.MetricRegistry;

import ru.yandex.bolts.function.Function;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.sidecars_updater.util.TStageAndDuId;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;

public abstract class StaticStatistics<T extends Number> implements Statistics {
    public static final String STATISTICS_NAME_PREFIX = "stat_";
    public static final String DEPLOY_UNIT_METRIC_SUFFIX = ".deployUnit";
    public static final String STAGE_METRIC_SUFFIX = ".stage";
    protected final String name;
    protected final Function<TStageAndDuId, T> deployUnitMetric;
    protected final Function<Collection<T>, T> stageMetric;
    protected final StatisticsMode statisticsMode;


    protected StaticStatistics(String name, Function<TStageAndDuId, T> deployUnitMetric,
                               Function<Collection<T>, T> stageMetric, StatisticsMode statisticsMode) {
        this.name = name;
        this.deployUnitMetric = deployUnitMetric;
        this.stageMetric = stageMetric;
        this.statisticsMode = statisticsMode;
    }


    public void prepare(MetricRegistry metricRegistry) {
        if (isDUStatistics()) {
            doPrepare(metricRegistry, getDUStatName());
        }
        if (isStageStatistics()) {
            doPrepare(metricRegistry, getStageStatName());
        }
    }

    protected abstract void doPrepare(MetricRegistry metricRegistry, String fullStatName);

    public void update(MetricRegistry metricRegistry,
                       Map<String, YpObject<StageMeta, TStageSpec, TStageStatus>> stages) {
        StatisticsResults<T> statisticsResults = getStatResults(stages);
        if (isDUStatistics()) {
            doUpdate(metricRegistry, getDUStatName(), statisticsResults.duStatistics);
        }
        if (isStageStatistics()) {
            doUpdate(metricRegistry, getStageStatName(), statisticsResults.stageStatistics);
        }
    }

    protected abstract void doUpdate(MetricRegistry metricRegistry, String statName, Map<String, T> metricResults);

    protected StatisticsResults<T> getStatResults(Map<String, YpObject<StageMeta, TStageSpec, TStageStatus>> stages) {
        Map<String, Map<String, T>> deployUnitMetricPerStage =
                stages.entrySet().stream().collect(Collectors.toMap(
                        Map.Entry::getKey,
                        stageEntry -> stageEntry.getValue().getSpec().getDeployUnitsMap()
                                .entrySet().stream().collect(Collectors.toMap(
                                        Map.Entry::getKey,
                                        duEntry -> deployUnitMetric.apply(
                                                new TStageAndDuId(stageEntry.getValue(), duEntry.getKey())
                                        ))))
                );

        Map<String, T> allDeployUnitMetricResults =
                deployUnitMetricPerStage.entrySet().stream()
                        .map(stageEntry -> stageEntry.getValue()
                                .entrySet().stream().collect(Collectors.toMap(
                                        (Map.Entry<String, T> duEntry) -> stageEntry.getKey() + "." + duEntry.getKey(),
                                        Map.Entry::getValue
                                ))).flatMap(map -> map.entrySet().stream())
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        Map<String, T> stageMetricResults =
                deployUnitMetricPerStage.entrySet().stream().collect(Collectors.toMap(
                        Map.Entry::getKey,
                        stageEntry -> stageMetric.apply(stageEntry.getValue().values())
                ));
        return new StatisticsResults<>(allDeployUnitMetricResults, stageMetricResults);
    }

    public boolean isStageStatistics() {
        return statisticsMode == StatisticsMode.ALL || statisticsMode == StatisticsMode.ONLY_STAGES;
    }

    public boolean isDUStatistics() {
        return statisticsMode == StatisticsMode.ALL || statisticsMode == StatisticsMode.ONLY_DU;
    }

    public String getName() {
        return name;
    }

    public String getStageStatName() {
        return STATISTICS_NAME_PREFIX + name + STAGE_METRIC_SUFFIX;
    }

    public Function<Collection<T>, T> getStageMetric() {
        return stageMetric;
    }

    public String getDUStatName() {
        return STATISTICS_NAME_PREFIX + name + DEPLOY_UNIT_METRIC_SUFFIX;
    }

    public Function<TStageAndDuId, T> getDeployUnitMetric() {
        return deployUnitMetric;
    }

    public StatisticsMode getStatisticsMode() {
        return statisticsMode;
    }

    public static class StatisticsResults<T> {
        final Map<String, T> duStatistics;
        final Map<String, T> stageStatistics;

        public StatisticsResults(Map<String, T> duStatistics, Map<String, T> stageStatistics) {
            this.duStatistics = duStatistics;
            this.stageStatistics = stageStatistics;
        }
    }

    public enum StatisticsMode {
        ALL,
        ONLY_STAGES,
        ONLY_DU
    }
}
