package ru.yandex.infra.sidecars_updater.statistics;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.codahale.metrics.UniformReservoir;

import ru.yandex.infra.sidecars_updater.util.TStageAndDuId;
import ru.yandex.infra.sidecars_updater.util.Utils;
import ru.yandex.qe.telemetry.metrics.yasm.YasmHistogram;
import ru.yandex.yp.client.api.TDeployUnitSpec;
import ru.yandex.yp.client.api.TStageSpec;


public class StatisticsRepository {

    public static final List<String> CLUSTERS = List.of("man", "myt", "sas", "vla", "iva");
    public static final String STAGES_WITH_MANY_DU = "stages_with_many_du";
    public static final String HAS_TVM = "has_tvm";
    public static final String MORE_THAN_2DU = "more_than_two_du";
    public static final String USE_LOGS = "use_logs";
    public static final String TRANSMIT_SYSTEM_LOGS = "transmit_system_logs";
    public static final String HAS_BOX_LIMITS = "has_box_limits";
    public static final String HAS_DOCKER = "has_docker";
    public static final String HAS_WORKLOAD_LIMITS = "has_workload_limits";
    public static final String WITHOUT_NETWORK_GUARANTEE = "without_network_guarantee";
    public static final String WITHOUT_HDD_IO_GUARANTEE = "without_hdd_io_guarantee";
    public static final String WITHOUT_HDD_IO_LIMITS = "without_hdd_io_limits";
    public static final String WITHOUT_SSD_IO_GUARANTEE = "without_ssd_io_guarantee";
    public static final String WITHOUT_SSD_IO_LIMITS = "without_ssd_io_limits";
    public static final String HAS_PORTO_METRICS = "has_porto_metrics";
    public static final String HAS_INHERIT_MISSED_MONITORING_LABELS = "has_inherit_missed_monitoring_labels";
    public static final String HAS_USER_SIGNALS = "has_user_signals";
    public static final String TOTAL = "total";
    public static final String CUSTOM_LOG_BROKER = "custom_log_broker";
    public static final String ALLOW_AUTO_UPDATE_SIDECARS = "allow_auto_update_sidecars";
    public static final String LOW_GUARANTEE_FOR_UNIFIED_AGENT = "low_guarantee_for_unified_agent";
    public static final String C_GROUP_FS_READ_ONLY = "c_group_fs_read_only";
    public static final String BOX_AMOUNT = "box_amount";
    public static final String SET_NETWORK_BANDWIDTH_GUARANTEE = "set_network_bandwidth_guarantee";
    public static final String SOX_STAGE = "sox_stage";
    public static final String SOX_STAGE_WITH_SAFE_SECRETS = "sox_stage_with_safe_secrets";
    public static final String SECRET_ENV_AND_CHILD_ONLY_ISOLATION = "secret_env_and_child_only_isolation";
    public static final String SECURE_RIGHTS_TO_STATIC_RESOURCES = "secure_rights_to_static_resources";
    public static final String CONFIGS_AND_BINARIES_VIA_VOLUME_MOUNTED_TO_BOX =
            "configs_and_binaries_via_volume_mounted_to_box";
    public static final String READ_ONLY_BOX_ROOTFS = "read_only_box_rootfs";
    public static final String USE_NON_PERSISTENT_VOLUME = "use_non_persistent_volume";
    public static final String THREAD_AMOUNT_IN_BOX = "thread_amount_in_box";
    public static final String USE_LAYERS_WITHOUT_DELETING = "use_layers_without_deleting";
    private static final String SEQUENTIAL_DEPLOYING_STRATEGY = "sequential_deploying_strategy";
    private static final String PARALLEL_DEPLOYING_STRATEGY = "parallel_deploying_strategy";
    private static final String DEFAULT_DEPLOYING_STRATEGY = "default_deploying_strategy";
    private static final String ALERTS_ENABLED = "alerts_enabled";
    private static final String CORE_DUMP_ENABLED = "core_dump_enabled";
    private static final String CORE_DUMP_AGGREGATION_ENABLED = "core_dump_aggregation_enabled";
    private static final String ENVIRONMENT_UNKNOWN = "environment_unknown";
    private static final String ENVIRONMENT_TESTING = "environment_testing";
    private static final String ENVIRONMENT_PRESTABLE = "environment_prestable";
    private static final String ENVIRONMENT_STABLE = "environment_stable";
    public static final String HISTOGRAM_STAT_SUFFIX = "ahhh";

    public static final String HDD = "hdd";
    public static final String SSD = "ssd";

    public static final String DISABLED = "disabled";
    public static final String USAGE = "usage";

    public static final Supplier<YasmHistogram> DEFAULT_YASM_HISTOGRAM_SUPPLIER =
            () -> new YasmHistogram(new UniformReservoir(), Set.of(HISTOGRAM_STAT_SUFFIX));

    public static Map<String, Integer> statisticsResults = new ConcurrentHashMap<>();
    public static Map<String, Map<Long, Set<String>>> statisticsGroupResults = new ConcurrentHashMap<>();
    private static final List<Statistics> logStatistics = List.of(
            new GlobalStatistics(USE_LOGS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseLogs)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(CUSTOM_LOG_BROKER,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> deployUnit.getLogbrokerConfig().hasCustomTopicRequest())),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(TRANSMIT_SYSTEM_LOGS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseSysLogs)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> limitStatistics = List.of(
            new GlobalStatistics(HAS_BOX_LIMITS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasBoxLimits)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(HAS_WORKLOAD_LIMITS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasLimitsOnWorkLoads)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(WITHOUT_HDD_IO_LIMITS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> Utils.isDeployUnitWithoutIoLimit(deployUnit, HDD))),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(WITHOUT_SSD_IO_LIMITS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> Utils.isDeployUnitWithoutIoLimit(deployUnit, SSD))),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> duAmountStatistics = List.of(
            new GlobalStatistics(STAGES_WITH_MANY_DU,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> true)),
                    Utils.countMetrics(),
                    Utils.countThresholdMetrics(2),
                    StaticStatistics.StatisticsMode.ONLY_STAGES),
            new GlobalStatistics(MORE_THAN_2DU,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> true)),
                    Utils.countMetrics(),
                    Utils.countThresholdMetrics(3),
                    StaticStatistics.StatisticsMode.ONLY_STAGES)
    );
    private static final List<Statistics> tvmStatistics = List.of(
            new GlobalStatistics(HAS_TVM,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(TDeployUnitSpec::hasTvmConfig)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> dockerStatistics = List.of(
            new GlobalStatistics(HAS_DOCKER,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasDocker)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> portoMetricsStatistics = List.of(
            new GlobalStatistics(HAS_PORTO_METRICS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(TDeployUnitSpec::getCollectPortometricsFromSidecars)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> monitoringStatistics = List.of(
            new GlobalStatistics(HAS_INHERIT_MISSED_MONITORING_LABELS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasInheritMissedMonitoringLabels)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(ALERTS_ENABLED,
                    Utils.onlyStageSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasAlertsEnabled)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES)
    );
    private static final List<Statistics> coreDumpStatistics = List.of(
            new GlobalStatistics(CORE_DUMP_ENABLED,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitEnableCoreDumps)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(CORE_DUMP_AGGREGATION_ENABLED,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitEnableCoreDumpsAggregation)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> userSignalStatistics = List.of(
            new GlobalStatistics(HAS_USER_SIGNALS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasUserSignals)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> networkStatistics = List.of(
            new GlobalStatistics(SET_NETWORK_BANDWIDTH_GUARANTEE,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isSetNetworkBandwidthGuarantee)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> updateStatistics = List.of(
            new GlobalStatistics(ALLOW_AUTO_UPDATE_SIDECARS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitAllowAutoSidecarsUpdate)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> guaranteeStatistics = List.of(
            new GlobalStatistics(LOW_GUARANTEE_FOR_UNIFIED_AGENT,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasLowGuaranteeForUnifiedAgent)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(WITHOUT_NETWORK_GUARANTEE,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitWithoutNetworkGuarantee)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(WITHOUT_HDD_IO_GUARANTEE,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> Utils.isDeployUnitWithoutIoGuarantee(deployUnit, HDD))),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(WITHOUT_SSD_IO_GUARANTEE,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> Utils.isDeployUnitWithoutIoGuarantee(deployUnit, SSD))),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> cGroupFsStatistics = List.of(
            new GlobalStatistics(C_GROUP_FS_READ_ONLY,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasCGroupFsReadOnlyMode)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> totalStatistics = List.of(
            new GlobalStatistics(TOTAL,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(deployUnit -> true)),
                    Utils.reduceMetrics(1, (a, b) -> 1),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> histogramStatistics = List.of(
            new HistogramStatistics(BOX_AMOUNT,
                    Utils.onlyDuSpecMetrics(Utils::getBoxAmount),
                    Utils.countMetrics(),
                    DEFAULT_YASM_HISTOGRAM_SUPPLIER,
                    StaticStatistics.StatisticsMode.ALL),
            new InnerHistogramStatistics(THREAD_AMOUNT_IN_BOX,
                    Utils::getNotZeroThreadAmountsInBoxes,
                    DEFAULT_YASM_HISTOGRAM_SUPPLIER)
    );
    private static final List<Statistics> soxStatistics = List.of(
            new GlobalStatistics(SOX_STAGE,
                    Utils.onlyStageSpecMetrics(Utils.predicateToInt(Utils::isSoxStage)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES),
            new GlobalStatistics(SOX_STAGE_WITH_SAFE_SECRETS,
                    Utils.predicateToInt(Utils::isDeployUnitHasOnlySecretsWithSafeAccessPermissions),
                    Utils.andMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES)
    );
    private static final List<Statistics> secureStatistics = List.of(
            new GlobalStatistics(SECRET_ENV_AND_CHILD_ONLY_ISOLATION,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasSecretEnvAndChildOnlyIsolation)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(SECURE_RIGHTS_TO_STATIC_RESOURCES,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasSecureRightsToStaticResources)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(CONFIGS_AND_BINARIES_VIA_VOLUME_MOUNTED_TO_BOX,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasConfigsAndBinariesViaVolumeMountedToBox)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(READ_ONLY_BOX_ROOTFS,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitHasReadOnlyBoxRootfs)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL),
            new GlobalStatistics(USE_NON_PERSISTENT_VOLUME,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseNonPersistentVolume)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> layerStatistics = List.of(
            new GlobalStatistics(USE_LAYERS_WITHOUT_DELETING,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseLayersWithoutDeleting)),
                    Utils.orMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ALL)
    );
    private static final List<Statistics> deployStatistics = List.of(
            new GlobalStatistics(SEQUENTIAL_DEPLOYING_STRATEGY,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseSequentialDeployingMode)),
                    Utils.countMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_DU),
            new GlobalStatistics(PARALLEL_DEPLOYING_STRATEGY,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseParallelDeployingMode)),
                    Utils.countMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_DU),
            new GlobalStatistics(DEFAULT_DEPLOYING_STRATEGY,
                    Utils.onlyDuSpecMetrics(Utils.predicateToInt(Utils::isDeployUnitUseDefaultDeployingMode)),
                    Utils.countMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_DU)
    );
    private static final List<Statistics> deployEnvironment = List.of(
            new GlobalStatistics(ENVIRONMENT_UNKNOWN,
                    Utils.onlyStageSpecMetrics(Utils::countUnknownEnv),
                    Utils.andMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES
            ),
            new GlobalStatistics(ENVIRONMENT_TESTING,
                    Utils.onlyStageSpecMetrics(Utils.countEnvironmentByType(TStageSpec.TDeployUnitSettings.EDeployUnitEnvironment.TESTING)),
                    Utils.andMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES
            ),
            new GlobalStatistics(ENVIRONMENT_PRESTABLE,
                    Utils.onlyStageSpecMetrics(Utils.countEnvironmentByType(TStageSpec.TDeployUnitSettings.EDeployUnitEnvironment.PRESTABLE)),
                    Utils.andMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES
            ),
            new GlobalStatistics(ENVIRONMENT_STABLE,
                    Utils.onlyStageSpecMetrics(Utils.countEnvironmentByType(TStageSpec.TDeployUnitSettings.EDeployUnitEnvironment.STABLE)),
                    Utils.andMetrics(),
                    Utils.countMetrics(),
                    StaticStatistics.StatisticsMode.ONLY_STAGES
            )
    );
    private static final List<Statistics> clusterUsageStatistic = generateUsageStat();
    private static final List<Statistics> clusterDisabledStatistic = generateDisabledStat();
    private static final List<GroupStatistics<Integer>> groupDuSpecStatistics =
            generateGroupStatistics(unionStatistics(List.of(
                    logStatistics, limitStatistics, duAmountStatistics, tvmStatistics, dockerStatistics,
                    portoMetricsStatistics, monitoringStatistics, totalStatistics, userSignalStatistics,
                    networkStatistics, updateStatistics, guaranteeStatistics, cGroupFsStatistics,
                    soxStatistics, secureStatistics, layerStatistics,
                    deployStatistics, coreDumpStatistics, deployEnvironment,
                    clusterUsageStatistic, clusterDisabledStatistic
            )));
    private static final List<? extends Statistics> basicStatistics = unionStatistics(
            List.of(logStatistics, limitStatistics, duAmountStatistics, tvmStatistics, dockerStatistics,
                    portoMetricsStatistics, monitoringStatistics, totalStatistics, userSignalStatistics,
                    networkStatistics, histogramStatistics, updateStatistics, guaranteeStatistics,
                    cGroupFsStatistics, soxStatistics, secureStatistics, layerStatistics, deployStatistics,
                    coreDumpStatistics, deployEnvironment,
                    clusterUsageStatistic, clusterDisabledStatistic,
                    groupDuSpecStatistics
            )
    );

    public static List<? extends Statistics> getBasicStatistics() {
        return basicStatistics;
    }

    private static List<? extends Statistics> unionStatistics(List<List<? extends Statistics>> statisticsGroups) {
        List<Statistics> statistics = new ArrayList<>();
        statisticsGroups.forEach(statistics::addAll);
        return statistics;
    }

    private static List<GroupStatistics<Integer>> generateGroupStatistics(List<? extends Statistics> statistics) {
        return statistics.stream()
                .filter(stat -> stat instanceof StaticStatistics)
                .map(stat ->
                {
                    StaticStatistics<Integer> staticStat = (StaticStatistics<Integer>) stat;
                    return new GroupStatistics<>(
                            staticStat.getName(),
                            staticStat.getDeployUnitMetric(),
                            staticStat.getStageMetric(),
                            staticStat.getStatisticsMode());
                }).collect(Collectors.toList());
    }

    private static List<Statistics> generateDisabledStat() {
        List<Statistics> stats = new ArrayList<>();
        CLUSTERS.forEach(cluster -> stats.add(new GlobalStatistics(DISABLED + "_" + cluster,
                (TStageAndDuId tStageAndDuId) -> Utils.countDisabledCluster(tStageAndDuId, cluster),
                Utils.orMetrics(),
                Utils.countMetrics(),
                StaticStatistics.StatisticsMode.ALL)));
        return stats;
    }

    private static List<Statistics> generateUsageStat() {
        List<Statistics> stats = new ArrayList<>();
        CLUSTERS.forEach(cluster -> stats.add(new GlobalStatistics(USAGE + "_" + cluster,
                Utils.onlyDuSpecMetrics((TDeployUnitSpec spec) -> Utils.countClusterInDuSpec(spec, cluster)),
                Utils.orMetrics(),
                Utils.countMetrics(),
                StaticStatistics.StatisticsMode.ALL
        )));
        return stats;
    }
}
