package ru.yandex.infra.sidecars_updater.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.bolts.function.Function;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.sidecars_updater.statistics.StaticStatistics;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TDeployUnitSpec;
import ru.yandex.yp.client.api.TLogbrokerConfig;
import ru.yandex.yp.client.api.TMonitoringInfo;
import ru.yandex.yp.client.api.TMonitoringUnistatEndpoint;
import ru.yandex.yp.client.api.TMonitoringWorkloadEndpoint;
import ru.yandex.yp.client.api.TPodTemplateSpec;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.client.pods.ECgroupFsMountMode;
import ru.yandex.yp.client.pods.ELayerSourceFileStoragePolicy;
import ru.yandex.yp.client.pods.EResourceAccessPermissions;
import ru.yandex.yp.client.pods.ETransmitSystemLogs;
import ru.yandex.yp.client.pods.EVolumeCreateMode;
import ru.yandex.yp.client.pods.EVolumeMountMode;
import ru.yandex.yp.client.pods.EVolumePersistenceType;
import ru.yandex.yp.client.pods.TBox;
import ru.yandex.yp.client.pods.TFile;
import ru.yandex.yp.client.pods.TMountedVolume;
import ru.yandex.yp.client.pods.TPodAgentSpec;
import ru.yandex.yp.client.pods.TResource;
import ru.yandex.yp.client.pods.TResourceGang;
import ru.yandex.yp.client.pods.TWorkload;

import static ru.yandex.yp.client.pods.EResourceAccessPermissions.EResourceAccessPermissions_600;
import static ru.yandex.yp.client.pods.EResourceAccessPermissions.EResourceAccessPermissions_660;

public class Utils {

    public static final String DEPLOY = "deploy";
    public static final String DISABLED_CLUSTERS = "disabled_clusters";

    public static <T> T getOtherValue(Stream<T> values, T value) {
        return values.filter(e -> !e.equals(value)).findAny().orElseThrow();
    }

    public static <T> T getOtherValue(T[] values, T value) {
        return getOtherValue(Arrays.stream(values), value);
    }

    public static <T> T getOtherValue(Collection<T> values, T value) {
        return getOtherValue(values.stream(), value);
    }

    public static boolean isDeployUnitUseLogs(TDeployUnitSpec deployUnit) {
        if (deployUnit.getLogbrokerConfig().getSidecarBringupMode() == TLogbrokerConfig.ESidecarBringupMode.MANDATORY) {
            return true;
        }
        TPodTemplateSpec podSpec = getPodFromDeployUnit(deployUnit);

        if (podSpec.getSpec().getPodAgentPayload().getSpec().getWorkloadsList().stream()
                .anyMatch(TWorkload::getTransmitLogs)) {
            return true;
        }
        return isDeployUnitUseSysLogs(deployUnit);
    }

    public static boolean isDeployUnitUseSysLogs(TDeployUnitSpec deployUnit) {
        ETransmitSystemLogs transmitSystemLogs =
                getPodFromDeployUnit(deployUnit).getSpec()
                        .getPodAgentPayload().getSpec()
                        .getTransmitSystemLogsPolicy().getTransmitSystemLogs();
        return transmitSystemLogs == ETransmitSystemLogs.ETransmitSystemLogsPolicy_ENABLED;
    }

    public static boolean isDeployUnitHasLimitsOnWorkLoads(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec().getPodAgentPayload().getSpec().getWorkloadsList().stream()
                .map(workload -> workload.getStart().getComputeResources())
                .anyMatch(computeResources ->
                        computeResources.getVcpuLimit() != 0 || computeResources.getMemoryLimit() != 0 ||
                                computeResources.getAnonymousMemoryLimit() != 0 || computeResources.getIoLimitCount() != 0 ||
                                computeResources.getIoOpsLimitCount() != 0);
    }

    public static boolean isDeployUnitWithoutNetworkGuarantee(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec().getResourceRequests().getNetworkBandwidthGuarantee() <= 0;
    }

    public static boolean isDeployUnitWithoutIoGuarantee(TDeployUnitSpec deployUnit, String storageClass) {
        return getPodFromDeployUnit(deployUnit).getSpec().getDiskVolumeRequestsList().stream()
                .filter(d -> d.getStorageClass().equals(storageClass))
                .anyMatch(r -> r.getQuotaPolicy().getBandwidthGuarantee() <= 0);
    }

    public static boolean isDeployUnitWithoutIoLimit(TDeployUnitSpec deployUnit, String storageClass) {
        return getPodFromDeployUnit(deployUnit).getSpec().getDiskVolumeRequestsList().stream()
                .filter(d -> d.getStorageClass().equals(storageClass))
                .anyMatch(r -> r.getQuotaPolicy().getBandwidthLimit() <= 0);
    }

    public static boolean isDeployUnitHasInheritMissedMonitoringLabels(TDeployUnitSpec deployUnit) {
        TMonitoringInfo monitoringInfo = getPodFromDeployUnit(deployUnit).getSpec().getHostInfra().getMonitoring();
        return monitoringInfo.getWorkloadsList().stream().anyMatch(TMonitoringWorkloadEndpoint::getInheritMissedLabels) ||
                monitoringInfo.getUnistatsList().stream().anyMatch(TMonitoringUnistatEndpoint::getInheritMissedLabels);
    }

    public static boolean isDeployUnitHasUserSignals(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec().getHostInfra().getMonitoring().getPodAgent().getAddPodAgentUserSignals();
    }

    public static boolean isDeployUnitHasBoxLimits(TDeployUnitSpec deployUnit) {
        return getBoxes(deployUnit).stream()
                .map(TBox::getComputeResources)
                .anyMatch(computeResources -> computeResources.getVcpuLimit() != 0 || computeResources.getMemoryLimit() != 0 ||
                        computeResources.getAnonymousMemoryLimit() != 0 || computeResources.getIoOpsLimitCount() != 0 ||
                        computeResources.getIoLimitCount() != 0);
    }

    public static boolean isDeployUnitHasDocker(TDeployUnitSpec deployUnit) {
        return deployUnit.getImagesForBoxesMap().size() > 0;
    }

    public static boolean isDeployUnitEnableCoreDumps(TDeployUnitSpec deployUnit) {
        return deployUnit.getCoredumpConfigMap().size() > 0;
    }

    public static boolean isDeployUnitEnableCoreDumpsAggregation(TDeployUnitSpec deployUnit) {
        return deployUnit.getCoredumpConfigMap().values().stream()
                .anyMatch(v -> v.getCoredumpProcessor().getAggregator().getEnabled());
    }

    public static boolean isSetNetworkBandwidthGuarantee(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec().getResourceRequests().getNetworkBandwidthGuarantee() != 0;
    }

    public static boolean isDeployUnitAllowAutoSidecarsUpdate(TDeployUnitSpec deployUnit) {
        return deployUnit.getInfraComponents().getAllowAutomaticUpdates();
    }

    public static boolean isDeployUnitHasLowGuaranteeForUnifiedAgent(TDeployUnitSpec deployUnit) {
        if (!deployUnit.hasLogbrokerConfig() || !deployUnit.getLogbrokerConfig().hasPodAdditionalResourcesRequest()) {
            return false;
        }
        DataModel.TPodSpec.TResourceRequests resourceRequests =
                deployUnit.getLogbrokerConfig().getPodAdditionalResourcesRequest();
        return resourceRequests.getVcpuGuarantee() == 0 && resourceRequests.getVcpuLimit() == 0;
    }

    public static boolean isDeployUnitHasCGroupFsReadOnlyMode(TDeployUnitSpec deployUnit) {
        return getBoxes(deployUnit).stream().anyMatch(box -> box.getCgroupFsMountMode() == ECgroupFsMountMode.ECgroupFsMountMode_RO);
    }

    public static boolean isDeployUnitHasSecretEnvAndChildOnlyIsolation(TDeployUnitSpec deployUnit) {
        return deployUnit.getPatchersRevision() >= 7 && deployUnit.getSoxService();
    }

    public static boolean isDeployUnitHasSecureRightsToStaticResources(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec()
                .getPodAgentPayload().getSpec()
                .getResources().getStaticResourcesList()
                .stream().anyMatch(resource -> resource.getAccessPermissions() !=
                        EResourceAccessPermissions.EResourceAccessPermissions_UNMODIFIED);
    }

    public static boolean isDeployUnitHasConfigsAndBinariesViaVolumeMountedToBox(TDeployUnitSpec deployUnit) {
        TPodAgentSpec podAgent = getPodFromDeployUnit(deployUnit).getSpec()
                .getPodAgentPayload().getSpec();
        Set<String> mountedVolumeReadOnlyRefs = podAgent.getBoxesList().stream()
                .flatMap(box -> box.getVolumesList().stream()
                        .filter(mountedVolume -> mountedVolume.getMode() == EVolumeMountMode.EVolumeMountMode_READ_ONLY)
                        .map(TMountedVolume::getVolumeRef))
                .collect(Collectors.toSet());
        return podAgent.getVolumesList().stream().anyMatch(volume ->
                (volume.getStaticResourcesCount() != 0 ||
                        volume.getGeneric().getLayerRefsCount() != 0) &&
                        mountedVolumeReadOnlyRefs.contains(volume.getId())
        );
    }

    public static boolean isDeployUnitHasReadOnlyBoxRootfs(TDeployUnitSpec deployUnit) {
        return deployUnit.getPatchersRevision() >= 7 &&
                getBoxes(deployUnit).stream()
                        .anyMatch(box -> box.getRootfs().getCreateMode() == EVolumeCreateMode.EVolumeCreateMode_READ_ONLY);
    }

    public static boolean isDeployUnitUseNonPersistentVolume(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec()
                .getPodAgentPayload().getSpec()
                .getVolumesList().stream()
                .anyMatch(volume -> volume.getPersistenceType() == EVolumePersistenceType.EVolumePersistenceType_NON_PERSISTENT);
    }

    public static boolean isDeployUnitUseLayersWithoutDeleting(TDeployUnitSpec deployUnit) {
        TResourceGang resources = getPodFromDeployUnit(deployUnit).getSpec()
                .getPodAgentPayload().getSpec()
                .getResources();
        return resources
                .getLayersList().stream()
                .anyMatch(layer -> layer.getLayerSourceFileStoragePolicy() ==
                        ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP) ||
                resources.getDefaultLayerSourceFileStoragePolicy() ==
                        ELayerSourceFileStoragePolicy.ELayerSourceFileStoragePolicy_KEEP;
    }

    public static boolean isDeployUnitHasOnlySecretsWithSafeAccessPermissions(TStageAndDuId tStageAndDuId) {
        List<TResource> staticResources = getPodFromDeployUnit(tStageAndDuId.getDuSpec()).getSpec()
                .getPodAgentPayload().getSpec()
                .getResources()
                .getStaticResourcesList().stream()
                .filter(resource -> resource.getFiles().getFilesList().stream().anyMatch(TFile::hasSecretData))
                .collect(Collectors.toList());
        return !staticResources.isEmpty() &&
                staticResources.stream()
                        .allMatch(resource ->
                                resource.getAccessPermissions() == EResourceAccessPermissions_660 ||
                                        resource.getAccessPermissions() == EResourceAccessPermissions_600
                        ) && tStageAndDuId.getStage().getSpec().getSoxService();
    }

    public static boolean isDeployUnitUseSequentialDeployingMode(TDeployUnitSpec deployUnit) {
        return isDeployUnitUseDeployingMode(deployUnit, TDeployUnitSpec.TDeploySettings.EDeployStrategy.SEQUENTIAL);
    }

    public static boolean isDeployUnitUseParallelDeployingMode(TDeployUnitSpec deployUnit) {
        return isDeployUnitUseDeployingMode(deployUnit, TDeployUnitSpec.TDeploySettings.EDeployStrategy.PARALLEL);
    }

    public static boolean isDeployUnitUseDefaultDeployingMode(TDeployUnitSpec deployUnit) {
        return !deployUnit.hasDeploySettings();
    }

    public static Collection<Integer> getNotZeroThreadAmountsInBoxes(TDeployUnitSpec deployUnit) {
        return getBoxes(deployUnit).stream()
                .map(box -> (int) box.getComputeResources().getThreadLimit())
                .filter(threadLimit -> threadLimit > 0)
                .collect(Collectors.toList());
    }

    public static int getBoxAmount(TDeployUnitSpec deployUnit) {
        return getBoxes(deployUnit).size();
    }

    private static boolean isDeployUnitUseDeployingMode(TDeployUnitSpec deployUnit,
                                                        TDeployUnitSpec.TDeploySettings.EDeployStrategy deployStrategy) {
        return deployUnit.hasDeploySettings() &&
                deployUnit.getDeploySettings().getDeployStrategy() == deployStrategy;
    }

    private static List<TBox> getBoxes(TDeployUnitSpec deployUnit) {
        return getPodFromDeployUnit(deployUnit).getSpec().getPodAgentPayload().getSpec().getBoxesList();
    }

    private static TPodTemplateSpec getPodFromDeployUnit(TDeployUnitSpec deployUnit) {
        return deployUnit.hasMultiClusterReplicaSet() ?
                deployUnit.getMultiClusterReplicaSet().getReplicaSet().getPodTemplateSpec() :
                deployUnit.getReplicaSet().getReplicaSetTemplate().getPodTemplateSpec();
    }

    public static boolean isSoxStage(TStageSpec stage) {
        return stage.getSoxService();
    }

    public static int countUnknownEnv(TStageSpec stage) {
        int totalDU = stage.getDeployUnitsCount();
        return totalDU - (int) stage.getDeployUnitSettingsMap().values().stream()
                .filter(du -> !TStageSpec.TDeployUnitSettings.EDeployUnitEnvironment.UNKNOWN.equals(du.getEnvironment()))
                .count();
    }

    public static Function<TStageSpec, Integer> countEnvironmentByType(
            TStageSpec.TDeployUnitSettings.EDeployUnitEnvironment environment) {
        return (stage) -> (int) stage.getDeployUnitSettingsMap().values().stream()
                .filter(du -> environment.equals(du.getEnvironment()))
                .count();
    }

    public static boolean isDeployUnitHasAlertsEnabled(TStageSpec stageSpec) {
        return stageSpec.getDeployUnitSettingsMap().entrySet().stream().anyMatch(e -> e.getValue().hasAlerting() && e.getValue().getAlerting().getState().equals(TStageSpec.TDeployUnitSettings.TAlerting.EState.ENABLED));
    }

    public static <T> StaticStatistics.StatisticsResults<T> getLabelStatResults(
            Map<String, YpObject<StageMeta, TStageSpec, TStageStatus>> stages,
            Function<Map.Entry<String, Map<String, YTreeNode>>, Map<String, T>> labelMetric,
            boolean isDuStat) {
        Map<String, T> metricResults =
                stages.entrySet().stream()
                        .flatMap(stageEntry -> labelMetric.apply(
                                Map.entry(
                                        stageEntry.getKey(),
                                        stageEntry.getValue().getLabels())
                        ).entrySet().stream())
                        .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        if (isDuStat) {
            return new StaticStatistics.StatisticsResults(metricResults, Map.of());
        }
        return new StaticStatistics.StatisticsResults(Map.of(), metricResults);
    }

    public static <T> Function<T, Integer> predicateToInt(Predicate<T> predicate) {
        return t -> predicate.test(t) ? 1 : 0;
    }

    public static Function<Collection<Integer>, Integer> reduceMetrics(Integer identity,
                                                                       BinaryOperator<Integer> reducer) {
        return collection -> collection.stream().reduce(identity, reducer);
    }

    public static Function<Collection<Integer>, Integer> countMetrics() {
        return reduceMetrics(0, Integer::sum);
    }

    public static Function<Collection<Integer>, Integer> countThresholdMetrics(int threshold) {
        return reduceMetrics(0, (a, b) -> a + (b >= threshold ? 1 : 0));
    }

    public static Function<Collection<Integer>, Integer> orMetrics() {
        return reduceMetrics(0, (a, b) -> Math.min(a + b, 1));
    }

    public static Function<Collection<Integer>, Integer> andMetrics() {
        return collection -> (collection.isEmpty() ? 0 : Collections.min(collection));
    }

    public static Function<Collection<Integer>, Integer> orStageMetrics() {
        return collection -> (collection.isEmpty() ? 1 : collection.stream().reduce(0, (a, b) -> Math.min(a + b, 1)));
    }

    public static <T> Function<TStageAndDuId, T> onlyStageSpecMetrics(
            Function<TStageSpec, T> metrics) {
        return (tStageAndDuId) -> metrics.apply(tStageAndDuId.getStage().getSpec());
    }

    public static int countClusterInDuSpec(TDeployUnitSpec spec, String cluster) {
        if (spec.hasMultiClusterReplicaSet()) {
            return (int) spec.getMultiClusterReplicaSet().getReplicaSet().getClustersList()
                    .stream()
                    .filter(it -> it.getCluster().equals(cluster))
                    .count();
        }
        return  (int) spec.getReplicaSet().getPerClusterSettingsMap().entrySet()
                .stream()
                .filter(it -> it.getKey().equals(cluster))
                .count();
    }

    public static List<String> getDisabledClusters(Map<String, YTreeNode> labels) {
        YTreeNode deployNode = labels.get(DEPLOY);
        if (deployNode == null) {
            return Collections.emptyList();
        }
        Optional<YTreeNode> disabledClustersNode = deployNode.mapNode().get(DISABLED_CLUSTERS);
        if (disabledClustersNode.isEmpty()) {
            return Collections.emptyList();
        }
        List<String> clusters = new ArrayList<>(disabledClustersNode.get().listNode().size());
        for (int i = 0; i < disabledClustersNode.get().listNode().size(); i++) {
            clusters.add(disabledClustersNode.get().listNode().getString(i));
        }
        return clusters;
    }

    public static int countDisabledCluster(TStageAndDuId tStageAndDuId, String cluster) {
        return getDisabledClusters(tStageAndDuId.getStageLabels()).stream()
                .filter(it -> it.equals(cluster))
                .mapToInt(it -> countClusterInDuSpec(tStageAndDuId.getDuSpec(), cluster))
                .sum();
    }

    public static <T> Function<TStageAndDuId, T> onlyDuSpecMetrics(Function<TDeployUnitSpec, T> func) {
        return tDeployUnitSpec -> func.apply(tDeployUnitSpec.getDuSpec());
    }
}
