package ru.yandex.infra.sidecars_updater.webauth;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.ListenableFuture;
import org.asynchttpclient.Response;

import ru.yandex.infra.sidecars_updater.sidecars.Sidecar;

public class WebAuthClientImpl implements WebAuthClient {
    private static final String WEBAUTH_SERVER = "https://webauth.yandex-team.ru";
    public static final String TEST = "test";
    private final AsyncHttpClient httpClient;
    private final List<String> userWhitelist;
    private final Map<Sidecar.Type, List<String>> sidecarOwners;
    private final boolean disabled;

    public WebAuthClientImpl(AsyncHttpClient httpClient, List<String> userWhitelist, Map<Sidecar.Type, List<String>> sidecarOwners, boolean disabled) {
        this.httpClient = httpClient;
        this.sidecarOwners = ImmutableMap.copyOf(sidecarOwners);
        this.disabled = disabled;
        if (this.disabled) {
            this.userWhitelist = List.of(TEST);
        } else {
            this.userWhitelist = ImmutableList.copyOf(userWhitelist);
        }
    }

    private ListenableFuture<Response> doGETRequest(String cookie) {
        return httpClient.executeRequest(httpClient
                .prepareGet(WEBAUTH_SERVER + "/auth_request")
                .addHeader("Cookie", cookie)
                .build()
        );
    }

    public CompletableFuture<String> getUserLogin(String cookie) {
        if (disabled) {
            return CompletableFuture.completedFuture(TEST);
        }
        return doGETRequest(cookie)
                .toCompletableFuture()
                .thenApply(response -> response.getHeader("X-Webauth-Login"));
    }

    @Override
    public boolean isUserWhitelisted(String login) {
        return userWhitelist.contains(login);
    }

    @Override
    public boolean isUserHasAccessToSidecar(String login, Sidecar.Type sidecar) {
        return sidecarOwners.getOrDefault(sidecar, Collections.emptyList()).contains(login);
    }

    @Override
    public boolean isDisabled() {
        return disabled;
    }
}
