package ru.yandex.infra.deploy;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.RepeatedTask;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.util.YsonUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpGetStatement;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpObjectUpdate;
import ru.yandex.yp.model.YpSetUpdate;
import ru.yandex.yp.model.YpTypedId;

import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class Deployer {
    private static final Logger LOG = LoggerFactory.getLogger(Deployer.class);

    static final String METRIC_RS_UPDATE_TIME = "rs_update_time_seconds";
    static final String METRIC_RS_READY_TIME = "rs_ready_time_seconds";
    static final String METRIC_DEPLOY_TIME = "deploy_time_seconds";
    static final String METRIC_STAGE_READY_TIME = "stage_ready_seconds";

    static class DeployerMetrics {
        private final Map<String, Long> metricRSUpdateTimePerCluster = new ConcurrentHashMap<>();
        private final Map<String, Long> metricRSReadyTimePerCluster = new ConcurrentHashMap<>();
        private final Map<String, Long> metricDeployTimePerCluster = new ConcurrentHashMap<>();
        private volatile Long metricStageReady;
    }

    private volatile DeployerMetrics previousIterationMetrics = new DeployerMetrics();
    private DeployerMetrics metrics = new DeployerMetrics();

    private final static int BOX_VARIABLE_INDEX_DEPLOY_RADAR_HOST = 0;
    private final static int BOX_VARIABLE_INDEX_REVISION = 1;

    private final static String RS_BOX_VARIABLE_PATH = "/spec/deploy_units/du/replica_set/replica_set_template/pod_template_spec/spec/pod_agent_payload/spec/boxes/0/env/%d/value/literal_env/value";
    private final static String MCRS_BOX_VARIABLE_PATH = "/spec/deploy_units/du/multi_cluster_replica_set/replica_set/pod_template_spec/spec/pod_agent_payload/spec/boxes/0/env/%d/value/literal_env/value";

    private final String stageCluster;
    private final String stageId;
    private final Map<String, YpRawObjectService> ypClientsPerCluster;
    private final Map<Long, Map<String, CompletableFuture<String>>> readyPerCluster = new ConcurrentHashMap<>();
    private final Map<Long, Map<String, CompletableFuture<?>>> rsUpdatedPerCluster = new ConcurrentHashMap<>();
    private final Map<Long, Map<String, CompletableFuture<?>>> rsReadyPerCluster = new ConcurrentHashMap<>();
    private final Map<Long, CompletableFuture<?>> stageReady = new ConcurrentHashMap<>();
    private final boolean isMultiClusterReplicaSet;
    private final String radarFqid;

    private Set<String> clusters;

    public Deployer(Duration updateInterval,
                    Duration mainLoopTimeout,
                    String stageCluster,
                    String stageId,
                    boolean isMultiClusterReplicaSet,
                    Map<String, YpRawObjectService> ypClientsPerCluster,
                    GaugeRegistry gaugeRegistry,
                    ScheduledExecutorService executor) {

        this.stageCluster = stageCluster;
        this.stageId = stageId;
        this.ypClientsPerCluster = ypClientsPerCluster;
        this.isMultiClusterReplicaSet = isMultiClusterReplicaSet;
        this.radarFqid = System.getenv("DEPLOY_POD_PERSISTENT_FQDN");

        ypClientsPerCluster.keySet().forEach(cluster -> {
            String prefix = cluster + "." + stageId + ".";
            if (!isMultiClusterReplicaSet) {
                gaugeRegistry.add(prefix + METRIC_RS_UPDATE_TIME, new GolovanableGauge<>(() -> getMetrics().metricRSUpdateTimePerCluster.get(cluster), "axxx"));
                gaugeRegistry.add(prefix + METRIC_RS_READY_TIME, new GolovanableGauge<>(() -> getMetrics().metricRSReadyTimePerCluster.get(cluster), "axxx"));
            }
            gaugeRegistry.add(prefix + METRIC_DEPLOY_TIME, new GolovanableGauge<>(() -> getMetrics().metricDeployTimePerCluster.get(cluster), "axxx"));
        });

        String prefix = stageCluster + "." + stageId + ".";
        if (isMultiClusterReplicaSet) {
            gaugeRegistry.add(prefix + METRIC_RS_UPDATE_TIME, new GolovanableGauge<>(() -> getMetrics().metricRSUpdateTimePerCluster.get(stageCluster), "axxx"));
            gaugeRegistry.add(prefix + METRIC_RS_READY_TIME, new GolovanableGauge<>(() -> getMetrics().metricRSReadyTimePerCluster.get(stageCluster), "axxx"));
        }
        gaugeRegistry.add(prefix + METRIC_STAGE_READY_TIME, new GolovanableGauge<>(() -> getMetrics().metricStageReady, "axxx"));

        RepeatedTask mainLoop = new RepeatedTask(this::mainLoop,
                updateInterval,
                mainLoopTimeout,
                executor,
                Optional.empty(),
                LOG,
                false);
        mainLoop.start();
    }

    public DeployerMetrics getMetrics() {
        return previousIterationMetrics;
    }

    public String getStageCluster() {
        return stageCluster;
    }

    public String getStageId() {
        return stageId;
    }

    public Set<String> getClusters() {
        return clusters;
    }

    public String getReplicaSetId() {
        return stageId + ".du";
    }

    public boolean isMultiClusterReplicaSet() {
        return isMultiClusterReplicaSet;
    }

    private CompletableFuture<?> mainLoop() {
        LOG.info("Starting deploy of stage {} on {}", stageId, stageCluster);

        previousIterationMetrics = metrics;
        metrics = new DeployerMetrics();

        readyPerCluster.clear();
        rsUpdatedPerCluster.clear();
        stageReady.clear();

        YpRawObjectService ypRawClient = ypClientsPerCluster.get(stageCluster);

        final YpTypedId ypTypedStageId = new YpTypedId(stageId, YpObjectType.STAGE);
        YpGetStatement getStageRevisionStatement = YpGetStatement.ysonBuilder(ypTypedStageId)
                .addSelector("/spec/revision")
                .addSelector(isMultiClusterReplicaSet ? "/spec/deploy_units/du/multi_cluster_replica_set/replica_set/clusters" :
                        "/spec/deploy_units/du/replica_set/per_cluster_settings")
                .build();

        Stopwatch stageDeployStopwatch = Stopwatch.createUnstarted();
        return ypRawClient.getObject(getStageRevisionStatement, payloads -> {
            long stageRevision = payloadToYson(payloads.get(0)).longValue();
            YTreeNode clustersNode = payloadToYson(payloads.get(1));
            clusters = isMultiClusterReplicaSet ?
                    clustersNode.asList().stream()
                            .map(node -> node.mapNode().get("cluster")
                                    .orElseThrow()
                                    .stringValue())
                            .collect(Collectors.toSet()) :
                    clustersNode.asMap().keySet();
            return stageRevision;
        })
        .thenCompose(stageRevision -> {
            LOG.info("Current {}.{} revision = {}", stageCluster, stageId, stageRevision);
            long newRevision = stageRevision + 1;

            String formatString = isMultiClusterReplicaSet ? MCRS_BOX_VARIABLE_PATH : RS_BOX_VARIABLE_PATH;
            YpObjectUpdate updateStatement = YpObjectUpdate.builder(ypTypedStageId)
                    .addSetUpdate(new YpSetUpdate(String.format(formatString, BOX_VARIABLE_INDEX_DEPLOY_RADAR_HOST),
                            YTree.stringNode(radarFqid),
                            YsonUtils::toYsonPayload))
                    .addSetUpdate(new YpSetUpdate(String.format(formatString, BOX_VARIABLE_INDEX_REVISION),
                            YTree.stringNode(String.valueOf(newRevision)),
                            YsonUtils::toYsonPayload))
                    .build();
            return ypRawClient.updateObject(updateStatement)
                    .thenApply(x -> newRevision);
        })
        .thenCompose(currentRevision -> {
            stageDeployStopwatch.start();
            LOG.info("{}.{} revision was updated to {}", stageCluster, stageId, currentRevision);

            Map<String, CompletableFuture<?>> rsUpdatedFutures = isMultiClusterReplicaSet ?
                    Map.of(stageCluster, new CompletableFuture<>()) :
                    clusters.stream().collect(Collectors.toMap(c -> c, c -> new CompletableFuture<>()));
            rsUpdatedPerCluster.put(currentRevision, rsUpdatedFutures);
            rsUpdatedFutures.forEach((cluster, readyFuture) ->
                    readyFuture.whenComplete((ignore, error) -> {
                        long elapsed = stageDeployStopwatch.elapsed(TimeUnit.SECONDS);
                        boolean isReady = error == null;
                        LOG.info("ReplicaSet update {}: cluster = {}, stage = {}, revision = {}, elapsed time = {}",
                                isReady ? "ready" : "timeout/failed", cluster, stageId, currentRevision, elapsed);
                        metrics.metricRSUpdateTimePerCluster.put(cluster, isReady ? elapsed : null);
                    })
            );

            Map<String, CompletableFuture<?>> rsReadyFutures = isMultiClusterReplicaSet ?
                    Map.of(stageCluster, new CompletableFuture<>()) :
                    clusters.stream().collect(Collectors.toMap(c -> c, c -> new CompletableFuture<>()));
            rsReadyPerCluster.put(currentRevision, rsReadyFutures);
            rsReadyFutures.forEach((cluster, readyFuture) ->
                    readyFuture.whenComplete((ignore, error) -> {
                        long elapsed = stageDeployStopwatch.elapsed(TimeUnit.SECONDS);
                        boolean isReady = error == null;
                        LOG.info("ReplicaSet ready {}: cluster = {}, stage = {}, revision = {}, elapsed time = {}",
                                isReady ? "ready" : "timeout/failed", cluster, stageId, currentRevision, elapsed);
                        metrics.metricRSReadyTimePerCluster.put(cluster, isReady ? elapsed : null);
                    })
            );

            Map<String, CompletableFuture<String>> workloadReadyFutures = clusters.stream()
                    .collect(Collectors.toMap(c -> c, c -> new CompletableFuture<>()));
            readyPerCluster.put(currentRevision, workloadReadyFutures);
            workloadReadyFutures.forEach((cluster, readyFuture) ->
                    readyFuture.whenComplete((pod, error) -> {
                        long elapsed = stageDeployStopwatch.elapsed(TimeUnit.SECONDS);
                        boolean isReady = error == null;
                        LOG.info("Workload {}: cluster = {}, stage = {}, revision = {}, pod = {}, elapsed time = {}",
                                isReady ? "ready" : "timeout/failed", cluster, stageId, currentRevision, pod, elapsed);
                        metrics.metricDeployTimePerCluster.put(cluster, isReady ? elapsed : null);
                    })
            );

            var stageReadyFuture = new CompletableFuture<>();
            stageReady.put(currentRevision, stageReadyFuture);
            stageReadyFuture
                    .whenComplete((ignore, error) -> {
                        long elapsed = stageDeployStopwatch.elapsed(TimeUnit.SECONDS);
                        boolean isReady = error == null;
                        LOG.info("Stage status {}: cluster {}, stage = {}, revision = {}, elapsed time = {}",
                                isReady ? "ready" : "timeout/failed", stageCluster, stageId, currentRevision, elapsed);
                        metrics.metricStageReady = isReady ? elapsed : null;
                    });

            return stageReadyFuture;
        });
    }

    public void workloadReady(String cluster, long revision, String pod) {
        LOG.debug("Received ready from: [{}] stage = {}, revision = {}, pod = {}", cluster, stageId, revision, pod);
        var futures = readyPerCluster.get(revision);
        if (futures != null) {
            var readyFuture = futures.get(cluster);
            if (readyFuture != null) {
                readyFuture.complete(pod);
            }
        }
    }

    public void rsIsReady(String cluster, long revision) {
        LOG.debug("ReplicaSet is ready: cluster = {}, stage = {}, revision = {}", cluster, stageId, revision);
        var futures = rsReadyPerCluster.get(revision);
        if (futures != null) {
            var readyFuture = futures.get(cluster);
            if (readyFuture != null) {
                readyFuture.complete(null);
            }
        }
    }

    public void rsUpdated(String cluster, long revision) {
        LOG.debug("ReplicaSet updated: cluster = {}, stage = {}, revision = {}", cluster, stageId, revision);
        var futures = rsUpdatedPerCluster.get(revision);
        if (futures != null) {
            var readyFuture = futures.get(cluster);
            if (readyFuture != null) {
                readyFuture.complete(null);
            }
        }
    }

    public void stageIsReady(long revision) {
        LOG.debug("Stage status is ready: cluster = {}, stage = {}, revision = {}", stageCluster, stageId, revision);
        var future = stageReady.get(revision);
        if (future != null) {
            future.complete(null);
        }
    }
}
