package ru.yandex.infra.deploy;

import java.io.File;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;

import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.ImmutableMap;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import com.typesafe.config.ConfigParseOptions;
import org.eclipse.jetty.server.Server;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.concurrent.DummyLeaderService;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.LeaderGaugeRegistry;
import ru.yandex.infra.controller.servlets.UnistatServlet;
import ru.yandex.infra.controller.util.ExitUtils;
import ru.yandex.infra.controller.util.YpUtils;
import ru.yandex.yp.YpRawObjectService;

import static ru.yandex.infra.controller.metrics.MetricUtils.buildMetricRegistry;
import static ru.yandex.infra.controller.servlets.ServerBuilder.buildServer;

public class Main {
    private static final Logger LOG = LoggerFactory.getLogger(Main.class);
    private static final String APPLICATION_DEFAULT_CONFIG_FILE_NAME = "application_defaults.conf";

    public static void main(String[] args) {
        try {
            doMain(args);
        } catch (Exception e) {
            LOG.error("Exception in main:", e);
            ExitUtils.gracefulExit(ExitUtils.EXCEPTION_MAIN);
        }
    }

    private static void doMain(String[] args) throws Exception {
        LOG.info("------------------------------------------------------------------------------------------------------");
        LOG.info("Starting new instance of Deploy Speed Radar...");

        Config config = ConfigFactory.parseFile(new File(args[0]), ConfigParseOptions.defaults().setAllowMissing(false))
                .withFallback(ConfigFactory.load(APPLICATION_DEFAULT_CONFIG_FILE_NAME));

        MetricRegistry metricRegistry = buildMetricRegistry(config.getConfig("metrics"));
        LeaderGaugeRegistry gaugeRegistry = new LeaderGaugeRegistry(metricRegistry, new DummyLeaderService(metricRegistry));

        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "deploy"));
        var ypClients = getYpClients(config, gaugeRegistry);
        Map<String, Deployer> deployers = getDeployers(config, gaugeRegistry, ypClients, executor);
        startServers(config, metricRegistry, deployers);
        var rsChecker = new ReplicaSetChecker(deployers.values(), ypClients,
                config.getDuration("engine.rs_checker.update_interval"),
                config.getDuration("engine.rs_checker.main_loop_timeout"));
        var stageChecker = new StageReadinessChecker(deployers, ypClients,
                config.getDuration("engine.stage_checker.update_interval"),
                config.getDuration("engine.stage_checker.main_loop_timeout"));
    }

    private static void startServers(Config config,
                                     MetricRegistry metricRegistry,
                                     Map<String, Deployer> deployers) throws Exception {

        Map<String, Map<String, HttpServlet>> servlets = ImmutableMap.of(
                "unistat", Map.of("unistat", new UnistatServlet(metricRegistry)),
                "api", Map.of("ready", new ReadyServlet(deployers))
        );

        var httpConfig = config.getConfig("http_server");

        var servers = httpConfig.root().keySet().stream()
                .map(entry -> buildServer(httpConfig.getConfig(entry), servlets.get(entry)))
                .collect(Collectors.toList());
        for (Server server: servers) {
            server.start();
        }
    }

    private static Map<String, YpRawObjectService> getYpClients(Config config, GaugeRegistry gaugeRegistry) {
        final Config ypConfig = config.getConfig("yp");
        final Config clusterConfigs = ypConfig.getConfig("clusters");
        return clusterConfigs.root().keySet().stream()
                .collect(Collectors.toMap(cluster -> cluster,
                        cluster -> YpUtils.ypRawClient(clusterConfigs.getConfig(cluster), ypConfig, gaugeRegistry, false)));
    }

    private static Map<String, Deployer> getDeployers(Config config,
                                                      GaugeRegistry gaugeRegistry,
                                                      Map<String, YpRawObjectService> ypClientsPerCluster,
                                                      ScheduledExecutorService executor) {

        final Duration updateInterval = config.getDuration("engine.update_interval");
        final Duration mainLoopTimeout = config.getDuration("engine.main_loop_timeout");

        Map<String, Deployer> result = new HashMap<>();

        config.getStringList("engine.stages").forEach(stageWithCluster -> {
            String[] s = stageWithCluster.split("\\.");
            result.put(stageWithCluster, new Deployer(updateInterval, mainLoopTimeout, s[0], s[1], false, ypClientsPerCluster, gaugeRegistry, executor));
        });
        config.getStringList("engine.mcrs_stages").forEach(stageWithCluster -> {
            String[] s = stageWithCluster.split("\\.");
            result.put(stageWithCluster, new Deployer(updateInterval, mainLoopTimeout, s[0], s[1], true, ypClientsPerCluster, gaugeRegistry, executor));
        });

        return result;
    }

}

