package ru.yandex.infra.deploy;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.RepeatedTask;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpGetManyStatement;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class ReplicaSetChecker {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicaSetChecker.class);

    private final Collection<Deployer> deployers;
    private final Map<String, YpRawObjectService> ypClientsPerCluster;
    private final Map<String, Deployer> deployersByReplicaSetId;

    public ReplicaSetChecker(Collection<Deployer> deployers,
                             Map<String, YpRawObjectService> ypClientsPerCluster,
                             Duration updateInterval,
                             Duration mainLoopTimeout) {
        this.deployers = deployers;
        this.ypClientsPerCluster = ypClientsPerCluster;

        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "rs_checker"));
        RepeatedTask mainLoop = new RepeatedTask(this::mainLoop,
                updateInterval,
                mainLoopTimeout,
                executor,
                Optional.empty(),
                LOG,
                false);

        deployersByReplicaSetId = deployers.stream()
                .collect(Collectors.toMap(Deployer::getReplicaSetId, s -> s));

        mainLoop.start();
    }

    private CompletableFuture<?> mainLoop() {

        Multimap<String, String> mcrsByCluster = HashMultimap.create();
        Multimap<String, String> replicaSetsByCluster = HashMultimap.create();

        deployers.forEach(stage -> {
            final String id = stage.getReplicaSetId();
            if (stage.isMultiClusterReplicaSet()) {
                mcrsByCluster.put(stage.getStageCluster(), id);
            } else {
                var clusters = stage.getClusters();
                if (clusters != null) {
                    clusters.forEach(cluster -> replicaSetsByCluster.put(cluster, id));
                }
            }
        });

        List<CompletableFuture<?>> futures = new ArrayList<>();
        addAllFutures(futures, mcrsByCluster, true);
        addAllFutures(futures, replicaSetsByCluster, false);
        return CompletableFuture.allOf(futures.toArray(CompletableFuture<?>[]::new));
    }

    private void addAllFutures(List<CompletableFuture<?>> futures, Multimap<String, String> rsByCluster, boolean isMCRS) {
        for (String cluster : rsByCluster.keys()) {
            CompletableFuture<?> future = getRequest(cluster, rsByCluster.get(cluster), isMCRS);
            futures.add(future);
        }
    }

    private CompletableFuture<?> getRequest(String cluster, Collection<String> ids, boolean isMCRS) {
        YpRawObjectService ypRawClient = ypClientsPerCluster.get(cluster);

        YpGetManyStatement statement = YpGetManyStatement.ysonBuilder(isMCRS ? YpObjectType.MULTI_CLUSTER_REPLICA_SET : YpObjectType.REPLICA_SET)
                .addSelector("/meta/id")
                .addSelector(isMCRS ? "/spec/revision" : "/spec/revision_id")
                .addSelector(isMCRS ? "/status/revision" : "/status/revision_id")
                .addSelector(isMCRS ? "/status/ready/status" : "/status/ready_condition/status")
                .addIds(new ArrayList<>(ids))
                .setIgnoreNonexistentObjects(true)
                .build();

        return ypRawClient.getObjects(statement, payloads -> {
            if (payloads.size() == 0) {
                return null;
            }
            String id = payloadToYson(payloads.get(0)).stringValue();

            YTreeNode specRevisionNode = payloadToYson(payloads.get(1));
            long specRevision = isMCRS ? specRevisionNode.longValue() : Long.parseLong(specRevisionNode.stringValue());
            var deployer = deployersByReplicaSetId.get(id);
            deployer.rsUpdated(cluster, specRevision);

            YTreeNode statusRevisionNode = payloadToYson(payloads.get(2));
            long statusRevision = isMCRS ? statusRevisionNode.longValue() : Long.parseLong(statusRevisionNode.stringValue());
            boolean ready = "true".equals(payloadToYson(payloads.get(3)).stringValue());
            if (ready) {
                deployer.rsIsReady(cluster, statusRevision);
            }

            return specRevision;
        });
    }
}
