package ru.yandex.infra.deploy;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.RepeatedTask;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpGetManyStatement;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class StageReadinessChecker {
    private static final Logger LOG = LoggerFactory.getLogger(StageReadinessChecker.class);

    private final Map<String, Deployer> deployers;
    private final Multimap<String, String> stagesByCluster = HashMultimap.create();
    private final Map<String, YpRawObjectService> ypClientsPerCluster;

    public StageReadinessChecker(Map<String, Deployer> deployers,
                                 Map<String, YpRawObjectService> ypClientsPerCluster,
                                 Duration updateInterval,
                                 Duration mainLoopTimeout) {
        this.deployers = deployers;
        this.ypClientsPerCluster = ypClientsPerCluster;

        deployers.values().forEach(deployer -> stagesByCluster.put(deployer.getStageCluster(), deployer.getStageId()));

        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "stage_readiness_checker"));
        RepeatedTask mainLoop = new RepeatedTask(this::mainLoop,
                updateInterval,
                mainLoopTimeout,
                executor,
                Optional.empty(),
                LOG,
                false);

        mainLoop.start();
    }

    private CompletableFuture<?> mainLoop() {

        List<CompletableFuture<?>> futures = new ArrayList<>();
        for (String cluster : stagesByCluster.keys()) {
            CompletableFuture<?> future = getRequest(cluster, stagesByCluster.get(cluster));
            futures.add(future);
        }

        return CompletableFuture.allOf(futures.toArray(CompletableFuture<?>[]::new));
    }

    private CompletableFuture<?> getRequest(String cluster, Collection<String> ids) {
        YpRawObjectService ypRawClient = ypClientsPerCluster.get(cluster);

        YpGetManyStatement statement = YpGetManyStatement.ysonBuilder(YpObjectType.STAGE)
                .addSelector("/meta/id")
                .addSelector("/status/deploy_units/du/target_revision")
                .addSelector("/status/deploy_units/du/ready/status")
                .addIds(new ArrayList<>(ids))
                .setIgnoreNonexistentObjects(true)
                .build();

        return ypRawClient.getObjects(statement, payloads -> {
            if (payloads.size() == 0) {
                return null;
            }
            String id = payloadToYson(payloads.get(0)).stringValue();
            long revision = payloadToYson(payloads.get(1)).longValue();
            boolean ready = "true".equals(payloadToYson(payloads.get(2)).stringValue());
            if (ready) {
                Deployer deployer = deployers.get(cluster + "." + id);
                deployer.stageIsReady(revision);
            }
            return revision;
        });
    }

}
